/* rep_hard:  print out size/inode/device numbers for the specified files

   Copyright (C) 2006-2025 by Brian Lindholm.  This file is part of the
   littleutils utility set.

   The rep_hard utility is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 3, or (at your option) any later
   version.

   The rep_hard utility is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
   or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
   more details.

   You should have received a copy of the GNU General Public License along with
   the littleutils.  If not, see <https://www.gnu.org/licenses/>. */


#include <config.h>

#if HAVE_INTTYPES_H
# include <inttypes.h>
#endif
#include <limits.h>
#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRING_H
# include <string.h>
#endif
#ifdef HAVE_SYS_STAT_H
# include <sys/stat.h>
#endif

#ifdef HAVE_UNISTD_H
# include <unistd.h>
# define OPTEND -1
#else
# define OPTEND EOF
#endif
#ifdef HAVE_GETOPT_H
# include <getopt.h>
#endif

#ifdef __MINGW32__
extern int getopt (int argc, char * const *argv, const char *optstring);
extern char *optarg;
extern int optind;
#endif

#ifdef DJGPP
/* speed up stat command for DJGPP */
unsigned short _djstat_flags = 63;
#endif

#ifndef PATH_MAX
# define PATH_MAX 4096
#endif

char *sizefmt_ps =
  (sizeof (off_t) <= sizeof (long) ? "%s\t%lu\t%d\t%d\n" : "%s\t%llu\t%d\t%d\n");


/* help function */

static void
help (FILE *where)
{
  fprintf (where,
    "rep_hard " PACKAGE_VERSION "\n"
    "usage: rep_hard [-f file_list] [-h(elp)] [-p(ipe)] file...\n");
}


/* print rep_hard function */

static void
printinfo (char *filename)
{
  struct stat file_stats;

  if (stat (filename, &file_stats))
    fprintf (stderr, "rep_hard error: can't determine info for %s\n", filename);
  else
    fprintf (stdout, sizefmt_ps, filename, file_stats.st_size,
      (unsigned int) file_stats.st_ino, (unsigned int) file_stats.st_dev);
}


/* main program */

int
main (int argc, char *argv[])
{
  FILE *infile;
  char filename[PATH_MAX], *listname, *newline, *rc;
  int argn, opt, use_file, use_pipe, verbose;

  /* parse options */

  listname = "";
  use_file = 0;
  use_pipe = 0;
  while ((opt = getopt (argc, argv, "f:hp")) != OPTEND)
    switch (opt)
      {
      case 'f':
        use_file = 1;
        listname = optarg;
        break;
      case 'h':
        help (stdout);
        return (0);
      case 'p':
        use_pipe = 1;
        break;
      case '?':
        help (stderr);
        return (1);
      }

  /* finalize options */

  if ((optind == argc) && (use_file == 0) && (use_pipe == 0))
    {
      help (stdout);
      return (0);
    }
  if (verbose == 0)
    {
      if (((argc - optind) != 1) || use_file || use_pipe)
        verbose = 1;
      else
        verbose = -1;
    }

  /* process files in listed in file specified by -f option */

  if (use_file)
    {
      infile = fopen (listname, "r");
      if (infile == NULL)
        fprintf (stderr, "rep_hard error: can't open %s!\n", listname);
      else
        {
          while (!feof (infile))
            {
              rc = fgets (filename, PATH_MAX - 1, infile);
              if (rc != NULL)
                {
                  newline = strchr (filename, '\n');
                  if (newline != NULL)
                    *newline = '\0';
                  if (strlen (filename) != 0)
                    printinfo (filename);
                }
            }
          (void) fclose (infile);
        }
    }

  /* process files listed on stdin (i.e., the -p option) */

  if (use_pipe)
    while (!feof (stdin))
      {
        rc = fgets (filename, PATH_MAX - 1, stdin);
        if (rc != NULL)
          {
            newline = strchr (filename, '\n');
            if (newline != NULL)
              *newline = '\0';
            if (strlen (filename) != 0)
              printinfo (filename);
          }
      }

  /* process files given in the argument list */

  for (argn = optind; argn < argc; argn++)
    printinfo (argv[argn]);

  /* indicate successful finish */

  return (0);
}
