/*
   BLAKE2 reference source code package - b2sum tool

   Copyright 2012, Samuel Neves <sneves@dei.uc.pt>.  You may use this under the
   terms of the CC0, the OpenSSL Licence, or the Apache Public License 2.0, at
   your option.  The terms of these licenses can be found at:

   - CC0 1.0 Universal : https://creativecommons.org/publicdomain/zero/1.0
   - OpenSSL license   : https://www.openssl.org/source/license.html
   - Apache 2.0        : https://www.apache.org/licenses/LICENSE-2.0

   More information about the BLAKE2 hash function can be found at
   https://blake2.net.

   Modifications for portability and partial file reads by Brian Lindholm,
   2020-2022.
*/

#include <config.h>

#ifdef HAVE_STDIO_H
# include <stdio.h>
#endif
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif
#ifdef HAVE_STRINGS_H
# include <string.h>
#endif

#include "blake2.h"

int blake2b_stream( FILE *stream, void *resstream, size_t outbytes, off_t readbytes )
{
  int ret = -1;
  size_t sum, n, target;
  blake2b_state S[1];
  static const size_t buffer_length = 32768;
  uint8_t *buffer = ( uint8_t * )malloc( buffer_length );
  off_t total = 0;

  if( !buffer ) return -1;

  blake2b_init( S, outbytes );
  if ( readbytes == 0 )
  {
    sum = 0;
    goto final_process;
  }

  while( 1 )
  {
    if ( ( readbytes < 0 ) || ( ( readbytes - total ) > (off_t) buffer_length ) )
      target = (size_t) buffer_length;
    else
      target = (size_t) ( readbytes - total );
    sum = 0;

    while( 1 )
    {
      if( ( (off_t) sum >= target ) || feof( stream ) )
        goto final_process;

      n = fread( buffer + sum, 1, target - sum, stream );
      sum += n;

      if( buffer_length == sum )
        break;

      if( 0 == n )
      {
        if( ferror( stream ) )
          goto cleanup_buffer;

        goto final_process;
      }
    }
    total += (off_t) sum;

    blake2b_update( S, buffer, buffer_length );
  }

final_process:;

  if( sum > 0 ) blake2b_update( S, buffer, sum );

  blake2b_final( S, resstream, outbytes );
  ret = 0;
cleanup_buffer:
  free( buffer );
  return ret;
}
