/*
 * disk/soft_touch.c
 *
 * Custom format as used on Platou from Soft Touch for Kingsoft.
 *
 * Written in 2023 by Keith Krellwitz
 *
 * RAW TRACK LAYOUT:
 *  u32 0x44894489  Sync
 *  u32 0xaaaaaaaa
 *  u32 dat[ti->len/4]
 *  u32 raw value - could not find any significance
 * 
 * It does not appear that there is a checksum
 *
 * TRKTYP_platou data layout:
 *  u8 sector_data[6000]
 * 
 */

#include <libdisk/util.h>
#include <private/disk.h>

static const uint16_t platou_crcs[];

static void *platou_write_raw(
    struct disk *d, unsigned int tracknr, struct stream *s)
{
    struct track_info *ti = &d->di->track[tracknr];

    while (stream_next_bit(s) != -1) {

        uint32_t raw[2], dat[ti->len/4+1];
        unsigned int i;
        char *block;

        if (tracknr != 2 && tracknr != 3 && tracknr != 4  && tracknr != 6)
            goto fail;

        if (s->word != 0x44894489)
            continue;
        ti->data_bitoff = s->index_offset_bc - 31;

        stream_start_crc(s);
        if (stream_next_bits(s, 32) == -1)
            goto fail;
        if (s->word != 0xaaaaaaaa)
            continue;

        for (i = 0; i < ti->len/4; i++) {
            if (stream_next_bytes(s, raw, 8) == -1)
                goto fail;
            mfm_decode_bytes(bc_mfm_even_odd, 4, raw, &dat[i]);
        }

        if (platou_crcs[tracknr-2] != s->crc16_ccitt)
            continue;

        // get last raw u32 and pass it via dat
        if (stream_next_bits(s, 32) == -1)
            goto fail;
        dat[ti->len/4] = s->word;

        stream_next_index(s);
        block = memalloc(ti->len+4);
        memcpy(block, dat, ti->len+4);
        set_all_sectors_valid(ti);
        ti->total_bits = s->track_len_bc;
        return block;
    }

fail:
    return NULL;
}

static void platou_read_raw(
    struct disk *d, unsigned int tracknr, struct tbuf *tbuf)
{
    struct track_info *ti = &d->di->track[tracknr];
    uint32_t *dat = (uint32_t *)ti->dat, sum;
    unsigned int i;

    tbuf_bits(tbuf, SPEED_AVG, bc_raw, 32, 0x44894489);
    tbuf_bits(tbuf, SPEED_AVG, bc_raw, 32, 0xaaaaaaaa);

    for (i = sum = 0; i < ti->len/4; i++) {
        tbuf_bits(tbuf, SPEED_AVG, bc_mfm_even_odd, 32, be32toh(dat[i]));
        sum += be32toh(dat[i]);
    }
    tbuf_bits(tbuf, SPEED_AVG, bc_raw, 32, dat[ti->len]);
}

struct track_handler platou_handler = {
    .bytes_per_sector = 6000,
    .nr_sectors = 1,
    .write_raw = platou_write_raw,
    .read_raw = platou_read_raw
};

static const uint16_t platou_crcs[] = {
    0x3ee3, 0x250d, 0xd989, 0x0000, 0x1185
};



/*
 * Custom format as used on several Soft Touch releases from Kingsoft
 * 
 * Evolution Cryser
 * Mind Force
 * Barney Mouse
 * Wizard's Castle
 * Sixiang
 *
 * None of the games have a data checksum
 * 
 * Written in 2023 by Keith Krellwitz
 *
 * RAW TRACK LAYOUT:
 *  u16 Sync :: Multiple syncs 0x2891, 0x2244, 0xa244, 0x9448, 0x4489
 *  u16 0 :: padding
 *  u32 dat[ti->len/4]
 *  u32 raw value - could not find any significance
 *
 * 
 * TRKTYP_evolution_cryser data layout:
 *  u8 sector_data[6300]
 * 
 * TRKTYP_mind_force data layout:
 *  u8 sector_data[6300]
 * 
 * TRKTYP_barney_mouse data layout:
 *  u8 sector_data[6300]
 * 
 *  Sync is always 0x4489
 * 
 * TRKTYP_wizards_castle_a data layout:
 *  u8 sector_data[6300]
 *
 * TRKTYP_wizards_castle_b data layout:
 *  u8 sector_data[6300]
 * 
 * TRKTYP_sixiang data layout:
 *  u8 sector_data[6300]
 */


static const uint16_t evolution_cryser_crcs[];
static const uint16_t evolution_cryser_syncs[];
static const uint16_t mind_force_crcs[];
static const uint16_t mind_force_syncs[];
static const uint16_t barney_mouse_crcs[];
static const uint16_t wizards_castle_a_crcs[];
static const uint16_t wizards_castle_a_syncs[];
static const uint16_t wizards_castle_b_crcs[];
static const uint16_t wizards_castle_b_syncs[];
static const uint16_t sixiang_crcs[];


static void *soft_touch_write_raw(
    struct disk *d, unsigned int tracknr, struct stream *s)
{
    struct track_info *ti = &d->di->track[tracknr];

    while (stream_next_bit(s) != -1) {
        uint32_t raw[2], dat[ti->len/4+1];
        char *block;
        unsigned int i;

        if (ti->type == TRKTYP_evolution_cryser) {
            if ((uint16_t)s->word != evolution_cryser_syncs[tracknr])
                continue;
        }
        else if (ti->type == TRKTYP_mind_force) {
            if ((uint16_t)s->word != mind_force_syncs[tracknr])
                continue;
        }
        else if (ti->type == TRKTYP_barney_mouse) {
            if ((uint16_t)s->word != 0x4489)
                continue;
        }
        else if (ti->type == TRKTYP_wizards_castle_a) {
            if ((uint16_t)s->word != wizards_castle_a_syncs[tracknr])
                continue;
        }
        else if (ti->type == TRKTYP_wizards_castle_b) {
            if ((uint16_t)s->word != wizards_castle_b_syncs[tracknr])
                continue;
        }
        else if (ti->type == TRKTYP_sixiang) {
            if ((uint16_t)s->word != wizards_castle_b_syncs[tracknr])
                continue;
        }
        ti->data_bitoff = s->index_offset_bc - 15;

        stream_start_crc(s);
        if (stream_next_bits(s, 32) == -1)
            goto fail;
        if (mfm_decode_word(s->word) != 0)
            continue;

        for (i = 0; i < ti->len/4; i++) {
            if (stream_next_bytes(s, raw, 8) == -1)
                goto fail;
            mfm_decode_bytes(bc_mfm_even_odd, 4, raw, &dat[i]);
        }

        if (ti->type == TRKTYP_evolution_cryser) {
            if(s->crc16_ccitt != evolution_cryser_crcs[tracknr])
                continue;
        } 
        else  if (ti->type == TRKTYP_mind_force) {
            if(s->crc16_ccitt != mind_force_crcs[tracknr] && 
                s->crc16_ccitt != mind_force_crcs[tracknr + 160])
                continue;
        }
        else  if (ti->type == TRKTYP_barney_mouse) {
            if(s->crc16_ccitt != barney_mouse_crcs[tracknr])
                trk_warn(ti, tracknr, "The checksum did not match, but may be a different version!");
        }
        else  if (ti->type == TRKTYP_wizards_castle_a) {
            if(s->crc16_ccitt != wizards_castle_a_crcs[tracknr])
                trk_warn(ti, tracknr, "The checksum did not match, but may be a different version!");
        }
        else  if (ti->type == TRKTYP_wizards_castle_b) {
            if(s->crc16_ccitt != wizards_castle_b_crcs[tracknr])
                continue;
        }
        else  if (ti->type == TRKTYP_sixiang) {
            if(s->crc16_ccitt != sixiang_crcs[tracknr])
                continue;
        }
        // get last raw u32 and pass it via dat
        if (stream_next_bits(s, 32) == -1)
            goto fail;
        dat[ti->len/4] = s->word;

        stream_next_index(s);
        block = memalloc(ti->len+4);
        memcpy(block, dat, ti->len+4);
        set_all_sectors_valid(ti);
        ti->total_bits = s->track_len_bc;
        return block;
    }

fail:
    return NULL;
}

static void soft_touch_read_raw(
    struct disk *d, unsigned int tracknr, struct tbuf *tbuf)
{
    struct track_info *ti = &d->di->track[tracknr];
    uint32_t *dat = (uint32_t *)ti->dat, sum;
    unsigned int i;

    if (ti->type == TRKTYP_evolution_cryser)
        tbuf_bits(tbuf, SPEED_AVG, bc_raw, 16, evolution_cryser_syncs[tracknr]);
    else if (ti->type == TRKTYP_mind_force)
        tbuf_bits(tbuf, SPEED_AVG, bc_raw, 16, mind_force_syncs[tracknr]);
    else if (ti->type == TRKTYP_barney_mouse)
        tbuf_bits(tbuf, SPEED_AVG, bc_raw, 16, 0x4489);
    else if (ti->type == TRKTYP_wizards_castle_a)
        tbuf_bits(tbuf, SPEED_AVG, bc_raw, 16, wizards_castle_a_syncs[tracknr]);
    else if (ti->type == TRKTYP_wizards_castle_b)
        tbuf_bits(tbuf, SPEED_AVG, bc_raw, 16, wizards_castle_b_syncs[tracknr]);
    else if (ti->type == TRKTYP_sixiang)
        tbuf_bits(tbuf, SPEED_AVG, bc_raw, 16, wizards_castle_b_syncs[tracknr]);
    tbuf_bits(tbuf, SPEED_AVG, bc_raw, 32, 0xaaaaaaaa);

    for (i = sum = 0; i < ti->len/4; i++) {
        tbuf_bits(tbuf, SPEED_AVG, bc_mfm_even_odd, 32, be32toh(dat[i]));
        sum += be32toh(dat[i]);
    }
   tbuf_bits(tbuf, SPEED_AVG, bc_mfm_even_odd, 32, dat[ti->len]);
}

struct track_handler evolution_cryser_handler = {
    .bytes_per_sector = 6300,
    .nr_sectors = 1,
    .write_raw = soft_touch_write_raw,
    .read_raw = soft_touch_read_raw
};

struct track_handler mind_force_handler = {
    .bytes_per_sector = 6300,
    .nr_sectors = 1,
    .write_raw = soft_touch_write_raw,
    .read_raw = soft_touch_read_raw
};

struct track_handler barney_mouse_handler = {
    .bytes_per_sector = 6300,
    .nr_sectors = 1,
    .write_raw = soft_touch_write_raw,
    .read_raw = soft_touch_read_raw
};

struct track_handler wizards_castle_a_handler = {
    .bytes_per_sector = 6300,
    .nr_sectors = 1,
    .write_raw = soft_touch_write_raw,
    .read_raw = soft_touch_read_raw
};

struct track_handler wizards_castle_b_handler = {
    .bytes_per_sector = 6300,
    .nr_sectors = 1,
    .write_raw = soft_touch_write_raw,
    .read_raw = soft_touch_read_raw
};

struct track_handler sixiang_handler = {
    .bytes_per_sector = 6300,
    .nr_sectors = 1,
    .write_raw = soft_touch_write_raw,
    .read_raw = soft_touch_read_raw
};

static const uint16_t evolution_cryser_crcs[] = {
    0x0000, 0x0000, 0xe018, 0x1635, 0x6a6c, 0x6de3, 0x2689, 0xe355, 
    0xf33a, 0x96ad, 0x6fee, 0x67b7, 0xb334, 0x1977, 0xf5c4, 0x7eb8, 
    0x5c1a, 0x1ebf, 0xb13d, 0x334a, 0xdde2, 0x4f9f, 0x4e5a, 0x0e53, 
    0xba49, 0x1c79, 0xa730, 0x073a, 0x57ff, 0xee84, 0x7777, 0xb48c, 
    0xec23, 0xe707, 0xf93b, 0x1deb, 0xaa54, 0xbb80, 0xcf5e, 0xdc1f, 
    0x5e25, 0x7d69, 0x7288, 0xde95, 0x289b, 0x2e89, 0x3075, 0x0f62, 
    0xe2dd, 0xead6, 0xd1b9, 0x36ea, 0xbbc1, 0x0686, 0xb830, 0x8374, 
    0x25c3, 0x93fc, 0x5a2e, 0xa08a, 0xeb47, 0x2f15, 0x28d1, 0x6440, 
    0x5fe0, 0x6095, 0x2d9e, 0x763c, 0xe625, 0x5da1, 0xe566, 0x0438, 
    0xdfc8, 0xcd44, 0x7b4b, 0x5da1, 0x3caf, 0x1050, 0x1ed3, 0x050d, 
    0xbd00, 0x5da1, 0x465a, 0xbbc7, 0x005f, 0xbac0, 0xaa3c, 0xb2d4, 
    0x1df6, 0x997d, 0x587c, 0x6176, 0xc717, 0x25f1, 0x5d12, 0x2601, 
    0x2d3f, 0x997d, 0x1ce2, 0x21bb, 0xacdb, 0x1c6e, 0x2e10, 0x3863, 
    0x874c, 0xb204, 0xe67c, 0xdbb0, 0x13f5, 0xa52f, 0xeb34, 0x2f77, 
    0x2289, 0xdca4, 0xc09c, 0xddf9, 0xf859, 0xdbb0, 0x99f0, 0x091f, 
    0x2e10, 0x150b, 0xce10, 0xb5b8, 0x3117, 0x6c9d, 0x51a0, 0xa52f, 
    0x17b6, 0xe8cd, 0x4493, 0x2375, 0xc81b, 0x054e, 0xd50d, 0xd665, 
    0xb720, 0xd278, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff
};

static const uint16_t evolution_cryser_syncs[] = {
    //0 - added to support probe all
    0x0000, 0x0000,
    //2
    0x2891, 0x2891, 
    //4
    0x2244, 0x2244, 0x2244, 0x2244, 0x2244, 0x2244, 0x2244, 0x2244,
    0x2244, 0x2244, 0x2244, 0x2244, 
    //16
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244,
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 
    //32
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244,
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 
    //48
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 
    //64
    0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 
    //70
    0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 
    //76
    0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 
    //82
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    //90
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    //98
    0x4489, 0x4489,
    //100
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489,
    //110
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    //118
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489,
    //128
    0x2244, 0x2244, 0x2244, 0x2244,
    //132
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244,
    //138 on not used
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244,
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244,
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244
};

static const uint16_t mind_force_crcs[] = {
    0x0000, 0x31d3, 0x252c, 0x8e8d, 0x621d, 0x7817, 0x52f8, 0xfba1,
    0xdb3a, 0xa89b, 0xc846, 0x558e, 0xd683, 0x270f, 0x82f9, 0xa2a3,
    0x7d47, 0x8ba3, 0x16aa, 0x761f, 0x4dca, 0x6865, 0x898f, 0x9d48,
    0x27fd, 0xe507, 0xa69e, 0x80bb, 0x11a0, 0x4c0b, 0x3fd9, 0x7531,
    0x3f13, 0x73b8, 0x4dae, 0x32e0, 0x6cd1, 0x7c84, 0xf16a, 0x5cfa,
    0x4628, 0xaf86, 0x51a2, 0xe3d3, 0xa667, 0x8eaf, 0x90c0, 0xafc7,
    0x2650, 0x08ef, 0xad71, 0xe401, 0x02f8, 0x85c9, 0x3cae, 0xa889,
    0x8a94, 0x5e39, 0x544b, 0xdc34, 0xb6e5, 0x5975, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,

    0x0000, 0x31d3, 0x252c, 0x8e8d, 0x621d, 0x7817, 0x52f8, 0xfba1,
    0xdb3a, 0xa89b, 0xc846, 0x558e, 0xd683, 0x270f, 0x82f9, 0xa2a3,
    0x7d47, 0x8ba3, 0x16aa, 0x761f, 0x4dca, 0x6865, 0x898f, 0x9d48,
    0x27fd, 0x504c, 0x55e1, 0x34c3, 0x496a, 0x9876, 0xbc6a, 0x7e38,
    0x5fbc, 0x73b8, 0x4dae, 0x32e0, 0x6cd1, 0x7c84, 0xf16a, 0x5cfa,
    0x5282, 0xaf86, 0x51a2, 0xe3d3, 0xa667, 0x8eaf, 0x90c0, 0xafc7,
    0x2650, 0x08ef, 0xad71, 0xe401, 0x02f8, 0x85c9, 0x3cae, 0xa889,
    0x8a94, 0x5e39, 0x544b, 0xdc34, 0xb6e5, 0x5975, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,

};

static const uint16_t mind_force_syncs[] = {
    0x0000, 0x2244, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x2891, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 
    0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244, 0xa244,
    0xa244, 0xa244, 0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 
    0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 0x9448, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff,
    0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff, 0xffff
};

static const uint16_t barney_mouse_crcs[] = {
    0x0000, 0x0028, 0x2f37, 0x9805, 0xf5ac, 0x9533, 0x3a1d, 0xa511, 
    0x1a44, 0xd6c7, 0xd1cc, 0x3977, 0xbaba, 0xc8f4, 0x15eb, 0x5e6e, 
    0xb105, 0xf53d, 0xeb5b, 0x0236, 0xea32, 0x0a18, 0x4287, 0x85c7, 
    0x2794, 0xc054, 0x7948, 0xbba4, 0x7ae0, 0x5a8b, 0x0234, 0xfa12, 
    0xe4e8, 0x20a5, 0x2c02, 0x4f2a, 0xf584, 0x4933, 0xd074, 0x3c71, 
    0xa4b2, 0xd8a9, 0xf8b7, 0x9035, 0xcf07, 0x6caa, 0x01a1, 0xd9c6, 
    0xa650, 0x95f9, 0x3b8b, 0x18b8, 0x0f55, 0xb79e, 0xaea6, 0x1898, 
    0xd4a7, 0x4a2b, 0x1f66, 0x02b4, 0x8bfc, 0x8b38, 0xd3d3, 0xbff6, 
    0x7438, 0xc041, 0xc46a, 0x7463, 0x03b9, 0xcdf7, 0xe2ba, 0xf4c6, 
    0xebe7, 0xbc4a, 0xda87, 0xe0da, 0x3b33, 0x72c6, 0x5129, 0xa6a8, 
    0xcafd, 0xea47, 0x171d, 0x3dc3, 0x0499, 0xa9a1, 0x3c7f, 0xd246, 
    0x0682, 0xa55d, 0xb89c, 0x51b4, 0x1c8b, 0x10bf, 0xc097, 0xbe0f, 
    0xdcc1, 0xe921, 0x0529, 0x61ad, 0x5a7c, 0xfd2f, 0xb8d2, 0x1550, 
    0xcbe3, 0x0e3e, 0xce1f, 0x2508, 0xe92f, 0x6e22, 0x418a, 0x95cf, 
    0x5a23, 0x32ff, 0x6a8d, 0x87d0, 0xc469, 0x7e5b, 0x5ef7, 0x0e1b, 
    0xff52, 0x1798, 0xc09b, 0x4c36, 0x0943, 0x2c05, 0x9ac0, 0x8453, 
    0x8289, 0x3ea4, 0x4941, 0xce66, 0x6953, 0xb998, 0xde26, 0xb4f3, 
    0x196e, 0x4218, 0x132f, 0x7be1, 0xa71b, 0xda2e, 0x5ed7, 0xfe71, 
    0xc428, 0xbeab, 0xbdec, 0x65ed, 0x5a95, 0x7d1c, 0x53a1, 0x0825, 
    0xa55f, 0xe2e1, 0x2303, 0x6008, 0x5a6d, 0x9ae6, 0x4534, 0x5cfb, 
    0xa67e, 0xd4fc
};


static const uint16_t wizards_castle_a_crcs[] = {
    0x0000, 0x0000, 0xf586, 0xa4dc, 0xdda6, 0x445a, 0xff7b, 0x4f4e, 
    0x15e2, 0x5f32, 0xeda4, 0xc62a, 0x4b7d, 0xd8e7, 0x5135, 0xf6ab, 
    0x26a3, 0x6653, 0x5ec0, 0x2713, 0x3354, 0xf53b, 0x17cd, 0xbb7e, 
    0xc9f1, 0xc3e9, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000
};

static const uint16_t wizards_castle_a_syncs[] = {
    0x0000, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x2244, 0x2244,
    0x2244, 0x2244, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489
};

static const uint16_t wizards_castle_b_crcs[] = {
    0x88f2, 0x1ccb, 0x91fd, 0xfbe0, 0x1a71, 0xb6ce, 0xaba8, 0x006b, 
    0xa9b8, 0xa04e, 0x3c5a, 0xa259, 0x7419, 0x19da, 0x160d, 0xeb54, 
    0x1175, 0x5c88, 0xa9bc, 0xaadf, 0x427f, 0x4d18, 0x0776, 0xaa45, 
    0xd4d6, 0x40e1, 0x3832, 0x0c9d, 0xcd97, 0xb70a, 0x91b0, 0x1295, 
    0xafc5, 0xc3bf, 0xca50, 0x3ae3, 0x3575, 0xc3bf, 0x4e59, 0x1cef, 
    0xe658, 0xc3bf, 0x5a6d, 0xc319, 0xd6b4, 0x4432, 0xbec2, 0x0b8c, 
    0xad95, 0x395a, 0x9494, 0xc4c5, 0x0fdb, 0xb76e, 0x5f1f, 0xf9af, 
    0x4a29, 0x2c00, 0x422a, 0xa52f, 0x99be, 0xd336, 0x1b3f, 0xc224, 
    0x7c73, 0xa52f, 0x88c9, 0x4df0, 0xb21c, 0x964d, 0x8780, 0xc7b1, 
    0x9c9c, 0xaa76, 0xc4ac, 0xa52f, 0xfed3, 0x0db6, 0x686d, 0xbfc1, 
    0xd70d, 0xa52f, 0x368d, 0x65b1, 0xe5bb, 0xd947, 0x4ac9, 0x4a76, 
    0xdd14, 0x5abe, 0x2124, 0xa52f, 0x1b91, 0xcced, 0x60f8, 0x37cc, 
    0x3148, 0x4c46, 0x498d, 0xf6a2, 0x99ca, 0xc73e, 0x5717, 0x7d94, 
    0xe475, 0x6431, 0x6688, 0xd2fd, 0xa914, 0xc244, 0xb358, 0x5ee9, 
    0x8ba4, 0xa52f, 0x2cf1, 0xc734, 0x1eda, 0x6aaf, 0xf559, 0x9706, 
    0x66c9, 0x117f, 0x21ff, 0x52ae, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
    0x0000, 0x0000
};

static const uint16_t wizards_castle_b_syncs[] = {
    0x2891, 0xa244, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489, 0x4489,
    0x4489, 0x4489
};

static const uint16_t sixiang_crcs[] = {
    0x0000, 0xc3c6, 0x701e, 0x0000, 0x8a03, 0x5b90, 0x8b2c, 0xa38c, 
    0x6510, 0x6f8d, 0x0a58, 0xf771, 0xe994, 0x22c1, 0xc262, 0x4df4, 
    0xe2c8, 0xf22b, 0x65ab, 0xa15c, 0xb857, 0xaab6, 0x1c5a, 0xcd8e, 
    0x4e9f, 0x5ebd, 0xac2c, 0x524f, 0xfa87, 0x4b48, 0xefec, 0xcd53, 
    0x5734, 0x650d, 0x111c, 0x9ce8, 0xbc63, 0x5fe9, 0xfc1e, 0x88c7, 
    0x1638, 0xec47, 0x2a4d, 0x1c56, 0x5e32, 0x0b20, 0xa487, 0x3c8d, 
    0xb8b6, 0x16c0, 0xc172, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x59ad, 0x3fc6, 0x0000, 
    0x7f85, 0x0000, 0x861d, 0x0000, 0x0000, 0x57b5, 0x0000, 0x0000, 
    0x0000, 0x0000, 0xafa0, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x5e29, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x9f97, 0x0000, 0x0000, 0xcde9, 0xe37f, 
    0x0000, 0x0273, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 
    0x0000, 0x0000, 0x0000, 0x0000, 0x9630, 0x0000, 0x0000, 0x0000, 
    0x66f5, 0x0000, 0x3c9b, 0x9ac4, 0x413a, 0x08a3, 0xca19, 0x2506
};


/*
 * Local variables:
 * mode: C
 * c-file-style: "Linux"
 * c-basic-offset: 4
 * tab-width: 4
 * indent-tabs-mode: nil
 * End:
 */
