#include "../../catch.hpp"

#include "../../../source/export/vgm/VgmExporter.h"
#include "../../../source/import/loader/SongLoader.h"
#include "../../../source/utils/MemoryBlockUtil.h"

#include <juce_core/juce_core.h>

#include <BinaryData.h>

namespace arkostracker 
{

// All the "expected" files are generated by the exporter. A bit of cheating, but they were tested with VGMPlay.
class TestTool
{
public:
    static constexpr auto changedOffset = 9;

    static juce::MemoryBlock exportToVgm(const char* songBinary, int songSize, bool compressed)
    {
        // Given.
        juce::MemoryInputStream inputStream(songBinary, static_cast<size_t>(songSize), false);
        SongLoader loader;
        const auto result = loader.loadSong(inputStream, "aks", false, nullptr);
        REQUIRE((result != nullptr));
        REQUIRE((result->errorReport != nullptr));
        REQUIRE(result->errorReport->isOk());
        const std::shared_ptr song = std::move(result->song);

        // When.
        VgmExporter vgmExporter(song, song->getFirstSubsongId(), compressed);
        const auto&[success, outputStream] = vgmExporter.performTask();

        // Then.
        REQUIRE(success);
        REQUIRE((outputStream != nullptr));

        return outputStream->getMemoryBlock();
    }
};

TEST_CASE("VgmExporter, harmless grenade loop at 4", "[VgmExporter]")
{
    const auto generatedVgm = TestTool::exportToVgm(BinaryData::At2TarghanAHarmlessGrenade_LoopAt4_aks, BinaryData::At2TarghanAHarmlessGrenade_LoopAt4_aksSize,
                                         false);
    const juce::MemoryBlock expectedMusicMemoryBlock(BinaryData::HarmlessLoopAt4_vgm, static_cast<size_t>(BinaryData::HarmlessLoopAt4_vgmSize));
    REQUIRE(MemoryBlockUtil::compare(generatedVgm, expectedMusicMemoryBlock));
}

TEST_CASE("VgmExporter, harmless grenade loop at 4, compressed", "[VgmExporter]")
{
    auto generatedVgm = TestTool::exportToVgm(BinaryData::At2TarghanAHarmlessGrenade_LoopAt4_aks, BinaryData::At2TarghanAHarmlessGrenade_LoopAt4_aksSize,
                                         true);
    const juce::MemoryBlock expectedMusicMemoryBlock(BinaryData::HarmlessLoopAt4_vgz, static_cast<size_t>(BinaryData::HarmlessLoopAt4_vgzSize));
#if JUCE_MAC
    generatedVgm[TestTool::changedOffset] = expectedMusicMemoryBlock[TestTool::changedOffset];       // On Mac, it seems the "compression method" changes. Fixes that for the test.
#endif
    REQUIRE(MemoryBlockUtil::compare(generatedVgm, expectedMusicMemoryBlock));
}

TEST_CASE("VgmExporter, last v8 9 channels, compressed", "[VgmExporter]")
{
    auto generatedVgm = TestTool::exportToVgm(BinaryData::At2TheLastV8_aks, BinaryData::At2TheLastV8_aksSize, true);
    const juce::MemoryBlock expectedMusicMemoryBlock(BinaryData::LastV86Channels_vgz, static_cast<size_t>(BinaryData::LastV86Channels_vgzSize));
#if JUCE_MAC
    generatedVgm[TestTool::changedOffset] = expectedMusicMemoryBlock[TestTool::changedOffset];       // On Mac, it seems the "compression method" changes. Fixes that for the test.
#endif
    REQUIRE(MemoryBlockUtil::compare(generatedVgm, expectedMusicMemoryBlock));
}

// ===================================================

TEST_CASE("Preliminary check, ok", "[VgmExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::At2TarghanAHarmlessGrenade_LoopAt4_aks,
                                        static_cast<size_t>(BinaryData::At2TarghanAHarmlessGrenade_LoopAt4_aksSize),false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "aks", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE(result->errorReport->isOk());
    const std::shared_ptr song = std::move(result->song);

    // When.
    const auto checkResult = VgmExporter::performPreliminaryCheck(*song, song->getFirstSubsongId());

    // Then.
    REQUIRE((checkResult.first == VgmExporter::PreliminaryCheck::ok));
}

TEST_CASE("Preliminary check, last v8 9 channels, too many PSGs", "[VgmExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::At2TheLastV8_aks, static_cast<size_t>(BinaryData::At2TheLastV8_aksSize),
                                        false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "aks", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE(result->errorReport->isOk());
    const std::shared_ptr song = std::move(result->song);

    // When.
    const auto checkResult = VgmExporter::performPreliminaryCheck(*song, song->getFirstSubsongId());

    // Then.
    REQUIRE((checkResult.first == VgmExporter::PreliminaryCheck::tooManyPsgs));
}

TEST_CASE("Preliminary check, different PSG frequencies", "[VgmExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::At2TheLastV86Channels_aks,
                                        static_cast<size_t>(BinaryData::At2TheLastV86Channels_aksSize),false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "aks", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE(result->errorReport->isOk());
    const std::shared_ptr song = std::move(result->song);

    // When.
    const auto checkResult = VgmExporter::performPreliminaryCheck(*song, song->getFirstSubsongId());

    // Then.
    REQUIRE((checkResult.first == VgmExporter::PreliminaryCheck::differentPsgFrequencies));
}

}   // namespace arkostracker
