#include "../../catch.hpp"

#include <BinaryData.h>

#include "../../../source/export/at3/SongExporter.h"
#include "../../../source/import/at3/At3SongImporter.h"
#include "../../../source/import/loader/SongLoader.h"
#include "../../../source/utils/XmlHelper.h"

namespace arkostracker 
{

// Effects, but has no speed/event tracks.
TEST_CASE("SongExporter, from Harmless Grenade", "[SongExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::At2TarghanAHarmlessGrenade_aks, BinaryData::At2TarghanAHarmlessGrenade_aksSize, false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "128", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE((result->errorReport->isOk()));
    auto song = std::move(result->song);

    song->resetDatesMs(123456);         // Else, comparison of dates will fail!

    // When.
    auto xmlSong = SongExporter::exportSong(*song);

    // Then.
    REQUIRE((xmlSong != nullptr));

    //REQUIRE((xmlSong->writeTo(juce::File("~/Documents/dev/cpc/tests/output.xml")));     // Nice for testing. The XML was generated by this.

    REQUIRE((XmlHelper::compare(*xmlSong, BinaryData::AT3HarmlessGrenade_xml, static_cast<size_t>(BinaryData::AT3HarmlessGrenade_xmlSize))));
}

TEST_CASE("SongExporter, from Harmless Grenade with some sfx not exported", "[SongExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::At2TarghanAHarmlessGrenade_aks, BinaryData::At2TarghanAHarmlessGrenade_aksSize, false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "128", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE((result->errorReport->isOk()));
    auto song = std::move(result->song);

    song->resetDatesMs(123456);         // Else, comparison of dates will fail!

    // Removes some sfxs.
    for (const auto instrumentIndex : { 2, 5 }) {
        song->performOnInstrument(song->getInstrumentId(instrumentIndex).getValue(), [] (Instrument& instrument) {
            auto& psgPart = instrument.getPsgPart();
            psgPart.setSoundEffectExported(false);
        });
    }

    // When.
    auto xmlSong = SongExporter::exportSong(*song);

    // Then.
    REQUIRE((xmlSong != nullptr));

    //REQUIRE((xmlSong->writeTo(juce::File("~/Documents/dev/cpc/tests/output.xml")));     // Nice for testing. The XML was generated by this.

    REQUIRE((XmlHelper::compare(*xmlSong, BinaryData::AT3HarmlessGrenadeWithoutSomeSfxs_xml,
        static_cast<size_t>(BinaryData::AT3HarmlessGrenadeWithoutSomeSfxs_xmlSize))));
}

// Not a lot of effects, but has speed/event tracks.
TEST_CASE("SongExporter, from dia end", "[SongExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::Targhan__DemoIzArt__End_Part_sks, BinaryData::Targhan__DemoIzArt__End_Part_sksSize, false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "sks", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE((result->errorReport->isOk()));
    auto song = std::move(result->song);

    song->resetDatesMs(123456);         // Else, comparison of dates will fail!

    // When.
    auto xmlSong = SongExporter::exportSong(*song);

    // Then.
    REQUIRE((xmlSong != nullptr));

    //REQUIRE((xmlSong->writeTo(juce::File("~/Documents/dev/cpc/tests/output.xml")));     // Nice for testing. The XML was generated by this.

    REQUIRE((XmlHelper::compare(*xmlSong, BinaryData::AT3DiaEnd_xml, static_cast<size_t>(BinaryData::AT3DiaEnd_xmlSize))));
}

TEST_CASE("SongExporter, event from Sarkboteur", "[SongExporter]")
{
    // Given.
    juce::MemoryInputStream inputStream(BinaryData::At2Sarkboteur_aks, BinaryData::At2Sarkboteur_aksSize, false);
    SongLoader loader;
    const auto result = loader.loadSong(inputStream, "aks", false, nullptr);
    REQUIRE((result != nullptr));
    REQUIRE((result->errorReport != nullptr));
    REQUIRE((result->errorReport->isOk()));
    auto song = std::move(result->song);

    song->resetDatesMs(123456);         // Else, comparison of dates will fail!

    // When.
    auto xmlSong = SongExporter::exportSong(*song);

    // Then.
    REQUIRE((xmlSong != nullptr));

    //REQUIRE((xmlSong->writeTo(juce::File("~/Documents/dev/cpc/tests/output.xml")));     // Nice for testing. The XML was generated by this.

    REQUIRE((XmlHelper::compare(*xmlSong, BinaryData::At3Sarkboteur_xml, static_cast<size_t>(BinaryData::At3Sarkboteur_xmlSize))));
}

TEST_CASE("SongExporter, read only", "[SongExporter]")
{
    // Imports the music with read-only.
    juce::MemoryInputStream inputStream(BinaryData::At3HocusPocusShorterWithReadOnly_xml, BinaryData::At3HocusPocusShorterWithReadOnly_xmlSize, false);
    At3SongImporter importer;
    REQUIRE((importer.doesFormatMatch(inputStream, "aks")));
    inputStream.setPosition(0);
    auto result = importer.loadSong(inputStream, ImportConfiguration());

    REQUIRE((result != nullptr));
    auto& song = result->song;
    auto& report = result->errorReport;
    REQUIRE((report != nullptr));
    REQUIRE((report->isOk()));
    REQUIRE((report->getWarningCount() == 0));
    REQUIRE((song != nullptr));

    song->resetDatesMs(1548457200000, 1712849293943);         // Else, comparison of dates will fail!

    // Then.
    // Exports the music with read-only. It should match.
    auto xmlSong = SongExporter::exportSong(*song);
    REQUIRE((xmlSong != nullptr));
    //REQUIRE((xmlSong->writeTo(juce::File("~/Documents/dev/cpc/tests/output.xml")));     // Nice for testing. The XML was generated by this.

    REQUIRE((XmlHelper::compare(*xmlSong, BinaryData::At3HocusPocusShorterWithReadOnly_xml, static_cast<size_t>(BinaryData::At3HocusPocusShorterWithReadOnly_xmlSize))));
}

TEST_CASE("SongExporter, named tracks", "[SongExporter]")
{
    // Imports the music with named tracks.
    juce::MemoryInputStream inputStream(BinaryData::At3HocusPocusShorterNamedTracks_xml, BinaryData::At3HocusPocusShorterNamedTracks_xmlSize, false);
    At3SongImporter importer;
    REQUIRE((importer.doesFormatMatch(inputStream, "aks")));
    inputStream.setPosition(0);
    auto result = importer.loadSong(inputStream, ImportConfiguration());

    REQUIRE((result != nullptr));
    auto& song = result->song;
    auto& report = result->errorReport;
    REQUIRE((report != nullptr));
    REQUIRE((report->isOk()));
    REQUIRE((report->getWarningCount() == 0));
    REQUIRE((song != nullptr));

    song->resetDatesMs(1548457200000, 1728247606063);         // Else, comparison of dates will fail!

    // Then.
    // Exports the music with named tracks. It should match.
    auto xmlSong = SongExporter::exportSong(*song);
    REQUIRE((xmlSong != nullptr));

    REQUIRE((XmlHelper::compare(*xmlSong, BinaryData::At3HocusPocusShorterNamedTracks_xml, static_cast<size_t>(BinaryData::At3HocusPocusShorterNamedTracks_xmlSize))));

    // Checks the names directly.
    const auto subsongId = song->getFirstSubsongId();
    song->performOnConstSubsong(subsongId, [&](const Subsong& subsong) {
        REQUIRE((subsong.getConstTrackRefFromPosition(0, 0).getName() == "Cool"));
        REQUIRE((subsong.getConstTrackRefFromPosition(0, 1).getName() == ""));
        REQUIRE((subsong.getConstTrackRefFromPosition(0, 2).getName() == "Stuff"));

        REQUIRE((subsong.getSpecialTrackRefFromPosition(0, true).getName() == "Speed0"));
        REQUIRE((subsong.getSpecialTrackRefFromPosition(0, false).getName() == "Link0"));

        REQUIRE((subsong.getSpecialTrackRefFromPosition(1, true).getName() == "Speed1"));
        REQUIRE((subsong.getSpecialTrackRefFromPosition(1, false).getName() == "Link1"));

        for (auto positionIndex = 1; positionIndex <= 8; ++positionIndex) {
            REQUIRE((subsong.getConstTrackRefFromPosition(positionIndex, 0).getName() == ""));
            REQUIRE((subsong.getConstTrackRefFromPosition(positionIndex, 1).getName() == ""));
            REQUIRE((subsong.getConstTrackRefFromPosition(positionIndex, 2).getName() == ""));
        }
    });
}

}   // namespace arkostracker
