#include "ExportFapDialog.h"

#include "../../../app/preferences/PreferencesManager.h"
#include "../../../controllers/MainController.h"
#include "../../../export/ExportConfiguration.h"
#include "../../../export/fap/FapExporter.h"
#include "../../../utils/FileExtensions.h"
#include "../../../utils/FileUtil.h"
#include "../../../utils/NumberUtil.h"
#include "../../../utils/StringUtil.h"
#include "../../components/FileChooserCustom.h"
#include "../../components/dialogs/SuccessOrErrorDialog.h"
#include "../../ProjectInfo.h"          // Generated file in the build folder.

namespace arkostracker 
{

ExportFapDialog::ExportFapDialog(MainController& pMainController, std::function<void()> pListener) noexcept:
        ModalDialog(juce::translate("Export to FAP"), 400, 170,
                    [&] { onExportButtonClicked(); },
                    [&] { onCancelButtonClicked(); },
                    true, true),
        songController(pMainController.getSongController()),
        listener(std::move(pListener)),
        subsongChooser(pMainController, [&](const std::pair<juce::String, Id>& /*nameAndId*/) { /** Nothing to do. */ }),
        versionLabel(juce::String(), juce::translate("FAP version: ") + projectinfo::fapVersion),
        backgroundTask(),
        fileToSaveTo(),
        dialog()
{
    setOkButtonText(juce::translate("Export"));
    setOkButtonWidth(70);
    setCancelButtonText(juce::translate("Close"));

    const auto bounds = getUsableModalDialogBounds();
    const auto left = bounds.getX();
    const auto top = bounds.getY();
    const auto width = bounds.getWidth();
    const auto margins = LookAndFeelConstants::margins;
    const auto labelsHeight = LookAndFeelConstants::labelsHeight;

    subsongChooser.setBounds(left, top, width, SubsongChooser::desiredHeight);
    versionLabel.setBounds(left, subsongChooser.getBottom() + margins, width, labelsHeight);

    addComponentToModalDialog(subsongChooser);
    addComponentToModalDialog(versionLabel);
}

void ExportFapDialog::onCancelButtonClicked() const noexcept
{
    listener();
}

void ExportFapDialog::onExportButtonClicked() noexcept
{
    // Opens the file picker.
    fileToSaveTo = FileChooserCustom::save(FileChooserCustom::Target::fap, FileExtensions::fapExtensionWithoutDot);
    if (fileToSaveTo.getFullPathName().isEmpty()) {
        return;     // Cancel.
    }

    const auto subsongId = subsongChooser.getSelectedSubsongId();
    const auto song = songController.getSong();
    const auto sourceConfiguration = SourceGeneratorConfiguration::buildZ80();      // Boilerplate.
    const auto baseLabel = juce::String();
    constexpr auto address = 0;
    constexpr auto encodeAllAddressesAsRelativeToSongStart = false;
    const ExportConfiguration exportConfiguration(sourceConfiguration, { subsongId }, baseLabel, address, SampleEncoderFlags::buildNoExport(),
                                            encodeAllAddressesAsRelativeToSongStart);

    // Creates the exporter, and the Task to perform it asynchronously.
    auto fapExporterTask = std::make_unique<FapExporter>(song, subsongId);

    backgroundTask = std::make_unique<BackgroundTaskWithProgress<std::unique_ptr<FapResult>>>(juce::translate("Exporting..."), *this,
                                                                                                     std::move(fapExporterTask));
    backgroundTask->performTask();
}


// BackgroundTaskListener method implementations.
// ======================================================

void ExportFapDialog::onBackgroundTaskFinished(const TaskOutputState taskOutputState, const std::unique_ptr<FapResult> result) noexcept
{
    backgroundTask->clear();

    // If canceled, nothing more to do.
    if (taskOutputState == TaskOutputState::canceled) {
        return;
    }

    // Pop-up if failure (not supposed to happen, though).
    if (taskOutputState != TaskOutputState::finished) {
        dialog = SuccessOrErrorDialog::buildForError(juce::translate("Unable to convert to song to FAP! Please report this."),
                                                            [&] { onDialogExit(false); });
        return;
    }

    // Ok, not on a background thread... Oh well.
    const auto& [fapData, bufferSize, playTimeInNops, registerCountToPlay, isR12Constant] = *result;
    const auto success = (bufferSize > 0) && FileUtil::saveMemoryBlockToFile(fileToSaveTo, fapData);
    if (success) {
        dialog = SuccessOrErrorDialog::buildForSuccess("Export to FAP finished! Buffer size = " + juce::String(bufferSize) +
            " (#" + NumberUtil::toHexFourDigits(bufferSize) + ").\nCPU in nops: " + juce::String(playTimeInNops)
            + ". Register count: " + juce::String(registerCountToPlay) + ". R12 constant? " + StringUtil::boolToYesOrNo(isR12Constant) + ".",
            [&] { onDialogExit(true); });
    } else {
        dialog = SuccessOrErrorDialog::buildForError("Unable to save the FAP file.", [&] { onDialogExit(); });
    }
}

// ======================================================

void ExportFapDialog::onDialogExit(const bool exit) noexcept
{
    dialog.reset();

    if (exit) {
        onCancelButtonClicked();
    }
}

}   // namespace arkostracker
