#pragma once

#include <memory>

#include "../SamplePlayInfo.h"
#include "ChannelOutputRegisters.h"

namespace arkostracker 
{

/**
 * This class holds data extracted from a ChannelPlayer after it has played one tick.
 * Also useful for generation, such as AKY, for example.
 */
class ChannelPlayerResults
{
public:
    /**
     * Constructor.
     * @param newPlayedNote present if a new note is played (that is, first tick on the line, and there is a note).
     * @param effectDeclared true if there is an effect (maybe with a note, or alone).
     * @param instrumentId the ID of the Instrument being played, if any.
     * @param channelOutputRegisters the registers for the Channel.
     * @param samplePlayInfo how to play a possible sample.
     * @param playedIndexInInstrument the cell index in a PSG Instrument, the sample index in a sample Instrument.
     */
    ChannelPlayerResults(OptionalInt newPlayedNote, bool effectDeclared, OptionalId instrumentId, std::unique_ptr<ChannelOutputRegisters> channelOutputRegisters,
                         SamplePlayInfo samplePlayInfo, int playedIndexInInstrument) noexcept;

    /** Constructor for an empty result (no sounds). */
    ChannelPlayerResults() noexcept;

    /** @return whether a new note is played. */
    bool isNewNotePlayed() const noexcept;
    /** @return a new note, or empty. */
    OptionalInt getNewPlayedNote() const noexcept;
    /** @return true if there is an effect only (maybe with a note, or alone). */
    bool isEffectDeclared() const noexcept;
    /** @return the ID of the Instrument being played, if any. */
    OptionalId getInstrumentId() const noexcept;
    /** @return the ChannelRegisters. */
    const ChannelOutputRegisters& getChannelOutputRegisters() const noexcept;

    /** @return the info about the possible sample to play. */
    const SamplePlayInfo& getSamplePlayInfo() const noexcept;
    /** The cell index in a PSG Instrument, the sample index in a sample Instrument. */
    int getPlayedIndexInInstrument() const noexcept;

    /** @return true if the given result is equal to the current one, regarding the PSG values (not the samples). */
    bool isPsgDataEqual(const ChannelPlayerResults& other) const noexcept;

private:
    OptionalInt newPlayedNote;                                          // Present if a new note is played.
    bool effectDeclared;                                                // True if there is an effect (maybe with a note, or alone).
    OptionalId instrumentId;                                            // The ID of the current Instrument being played, if any.
    std::unique_ptr<ChannelOutputRegisters> channelOutputRegisters;     // The registers generated by the Channel.

    SamplePlayInfo samplePlayInfo;
    int playedIndexInInstrument;
};

}   // namespace arkostracker
