#include "SfxExporter.h"

#include <utility>

#include "../../player/channel/ChannelPlayer.h"
#include "../../song/Song.h"
#include "../../utils/BitNumber.h"
#include "../sourceGenerator/SourceGenerator.h"

namespace arkostracker
{

SfxExporter::SfxExporter(const Song& pSong, ExportConfiguration pExportConfiguration) noexcept :
        song(pSong),
        exportConfiguration(std::move(pExportConfiguration))
{
}

std::pair<bool, std::unique_ptr<SongExportResult>> SfxExporter::performTask() noexcept
{
    auto exportResult = std::make_unique<SongExportResult>();

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    const auto baseLabel = getSoundEffectBaseLabel();

    sourceGenerator.setPrefixForDisark(baseLabel);
    const auto songName = song.getName();
    sourceGenerator.declareComment((songName.isEmpty() ? "" : (songName + ", ")) + "Sound effects, format v2.0.").addEmptyLine();
    sourceGenerator.declareComment("Generated by Arkos Tracker 3.").addEmptyLine();

    // Org?
    if (exportConfiguration.getAddress().isPresent()) {
        sourceGenerator.declareAddressChange(exportConfiguration.getAddress().getValue()).addEmptyLine();
    }

    sourceGenerator.declareLabel(baseLabel);
    sourceGenerator.declareExternalLabel(baseLabel).addEmptyLine();

    // Builds the list of the exported instruments.
    const auto sfxData = FirstNotePerInstrumentFinder::find(song);

    // Encodes a table of address for each Instrument. Contrary to AT2, non-exported effects are not encoded.
    sourceGenerator.declareComment("The sound effects, starting at 1.");
    sourceGenerator.declarePointerRegionStart();   // Note that we do not manage the "0" below differently for Disark. Who cares?
    auto sfxIndex = 0;
    for (const auto& [instrumentIndex, sfx] : sfxData) {
        if (sfx.canUseUsed()) {
            const auto labelToEncode = getSoundEffectLabel(instrumentIndex);
            sourceGenerator.declareWord(labelToEncode, "Sound effect " + juce::String(instrumentIndex) + " at index " + juce::String(sfxIndex + 1) + ".");

            ++sfxIndex;
        }
    }
    sourceGenerator.declarePointerRegionEnd();
    sourceGenerator.addEmptyLine();

    PlayerConfiguration playerConfiguration(false);

    // Encodes each Instrument that needs to be.
    sourceGenerator.declareByteRegionStart();
    for (const auto& [instrumentIndex, sfx] : sfxData) {
        encodeSoundEffect(instrumentIndex, sfx, sourceGenerator, playerConfiguration);
    }
    sourceGenerator.declareByteRegionEnd();
    sourceGenerator.declareEndOfFile();

    auto result = std::make_unique<SongExportResult>(std::move(songOutputStream), playerConfiguration);
    return { true, std::move(result) };
}

juce::String SfxExporter::getSoundEffectBaseLabel(const bool addEndSeparator) const noexcept
{
    const auto endSeparator = addEndSeparator ? juce::String("_") : juce::String();
    return exportConfiguration.getBaseLabel() + "SoundEffects" + endSeparator;
}

juce::String SfxExporter::getSoundEffectLabel(const int soundEffectIndex) const noexcept
{
    return getSoundEffectBaseLabel(true) + "Sound" + juce::String(soundEffectIndex);
}

juce::String SfxExporter::getSoundEffectLoopLabel(const int soundEffectIndex) const noexcept
{
    return getSoundEffectLabel(soundEffectIndex) + "_Loop";
}

void SfxExporter::encodeSoundEffect(int instrumentIndex, const FirstNotePerInstrumentFinder::ResultItem& sfxData,
                                    SourceGenerator& sourceGenerator, PlayerConfiguration& playerConfiguration) const noexcept
{
    // Don't encode if not needed.
    if (!sfxData.canUseUsed()) {
        return;
    }
    jassert(instrumentIndex != 0);      // Must be skipped!

    // Disark: we consider a Byte Region for all sound has already been declared.
    sourceGenerator.declareComment("Sound effect " + juce::String(instrumentIndex) + ".");
    sourceGenerator.declareLabel(getSoundEffectLabel(instrumentIndex));

    // Encodes the header, gets some instrument data.
    PsgPart psgPart;
    song.performOnConstInstrumentFromIndex(instrumentIndex, [&](const Instrument& instrument) {
        jassert(instrument.getType() == InstrumentType::psgInstrument);
        psgPart = instrument.getConstPsgPart();
    });

    const auto instrumentSpeed = psgPart.getSpeed();
    jassert(psgPart.isSoundEffectExported());
    sourceGenerator.declareByte(instrumentSpeed, "Speed").addEmptyLine();

    SfxDataProvider songDataProvider(song, psgPart);

    // Builds a ChannelPlayer with a note to play. We will "record" what is being played.
    constexpr auto channelIndex = 0;
    ChannelPlayer channelPlayer(songDataProvider, channelIndex);
    const auto note = sfxData.getFoundNote().getValue();
    jassert(sfxData.getFoundNote().isPresent());        // Mandatory!
    const auto cell = Cell::build(note, instrumentIndex);
    channelPlayer.postCell(CellToPlay(cell, channelIndex, {}));

    auto cellIndex = 0;
    const auto loop = psgPart.getMainLoop();
    const auto loopStartIndex = loop.getStartIndex();
    const auto endIndex = loop.getEndIndex();
    const auto isLooping = loop.isLooping();
    auto tick = 0;       // <= speed;
    while (cellIndex <= endIndex) {
        const auto isFirstTick = (tick == 0);
        channelPlayer.playStream(isFirstTick, true);

        // Only the tick 0 is worth encoding, the others are only duplicates.
        if (isFirstTick) {
            const auto channelResult = channelPlayer.getResults();
            encodeRegisters(sourceGenerator, channelResult->getChannelOutputRegisters(), cellIndex, instrumentIndex,
                            isLooping ? loopStartIndex : OptionalInt(), playerConfiguration);
            sourceGenerator.addEmptyLine();
        }

        if (++tick > instrumentSpeed) {
            tick = 0;
            ++cellIndex;
        }
    }

    // Encodes the end/loop.
    BitNumber endByte;
    endByte.injectBits("00");
    endByte.injectBool(true);       // Marks the end.
    endByte.injectBool(isLooping);
    sourceGenerator.declareByte(endByte.get(), "End of the sound effect.");
    if (isLooping) {
        sourceGenerator.declarePointerRegionStart();
        sourceGenerator.declareWord(getSoundEffectLoopLabel(instrumentIndex), "Loops here.");
        sourceGenerator.declarePointerRegionEnd();

        playerConfiguration.addFlag(PlayerConfigurationFlag::sfxLoopTo);
    }
    sourceGenerator.addEmptyLine();
}

void SfxExporter::encodeRegisters(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters, const int cellIndex,
                                  const int instrumentIndex, const OptionalInt loopIndex, PlayerConfiguration& playerConfiguration) const noexcept
{
    // Should the loop label be encoded? Yes if the Instrument loops here.
    if (loopIndex.isPresent() && (cellIndex == loopIndex.getValue())) {
        sourceGenerator.declareLabel(getSoundEffectLoopLabel(instrumentIndex));
    }

    // Encodes the registers according to the type of the sound line.
    switch (channelRegisters.getSoundType()) {
        case SoundType::noSoftwareNoHardware:
            encodeCellNoSoftwareNoHardware(sourceGenerator, channelRegisters, playerConfiguration);
            break;
        case SoundType::softwareOnly:
            encodeCellSoftwareOnly(sourceGenerator, channelRegisters, playerConfiguration);
            break;
        case SoundType::hardwareOnly:
            encodeCellHardwareOnly(sourceGenerator, channelRegisters, playerConfiguration);
            break;
        case SoundType::softwareAndHardware:
            encodeCellSoftwareAndHardware(sourceGenerator, channelRegisters, playerConfiguration);
            break;
        default:
            jassertfalse;       // Shouldn't happen.
            break;
    }
}

void SfxExporter::encodeCellNoSoftwareNoHardware(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters, PlayerConfiguration& playerConfiguration) noexcept
{
    const auto isNoise = channelRegisters.isNoise();
    const auto volume = channelRegisters.getVolume();
    jassert((volume >= 0) && (volume <= 15));

    BitNumber bitNumber;
    bitNumber.injectBits("00");
    bitNumber.injectBool(false);        // Not end of sound effect.
    bitNumber.injectNumber(volume, 4U);
    bitNumber.injectBool(isNoise);
    sourceGenerator.declareByte(bitNumber.get(), "No soft, no hard. Volume: " + juce::String(volume) + ".");

    encodeNoiseIfAny(sourceGenerator, channelRegisters);
    playerConfiguration.addFlag(isNoise, PlayerConfigurationFlag::sfxNoSoftNoHardNoise);
    playerConfiguration.addFlag(PlayerConfigurationFlag::sfxNoSoftNoHard);
}

void SfxExporter::encodeCellSoftwareOnly(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters, PlayerConfiguration& playerConfiguration) noexcept
{
    const auto isNoise = channelRegisters.isNoise();
    const auto volume = channelRegisters.getVolume();
    jassert((volume >= 0) && (volume <= 15));

    BitNumber bitNumber;
    bitNumber.injectBits("01");
    bitNumber.injectNumber(volume, 4U);
    bitNumber.injectBool(false);        // Ignored bit.
    bitNumber.injectBool(isNoise);
    sourceGenerator.declareByte(bitNumber.get(), "Soft only. Volume: " + juce::String(volume) + ".");

    encodeNoiseIfAny(sourceGenerator, channelRegisters);
    playerConfiguration.addFlag(isNoise, PlayerConfigurationFlag::sfxSoftOnlyNoise);
    playerConfiguration.addFlag(PlayerConfigurationFlag::sfxSoftOnly);

    // Encodes the software period.
    const auto softwarePeriod = channelRegisters.getSoftwarePeriod();
    sourceGenerator.declareWord(softwarePeriod, "Software period.");
}

void SfxExporter::encodeCellHardwareOnly(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters, PlayerConfiguration& playerConfiguration) noexcept
{
    encodeCellSharedHardwareOrSoftwareAndHardware(sourceGenerator, channelRegisters, true, playerConfiguration);
    playerConfiguration.addFlag(PlayerConfigurationFlag::sfxHardOnly);
}

void SfxExporter::encodeCellSoftwareAndHardware(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters, PlayerConfiguration& playerConfiguration) noexcept
{
    encodeCellSharedHardwareOrSoftwareAndHardware(sourceGenerator, channelRegisters, false, playerConfiguration);

    // Encodes the software period.
    const auto softwarePeriod = channelRegisters.getSoftwarePeriod();
    sourceGenerator.declareWord(softwarePeriod, "Software period.");

    playerConfiguration.addFlag(PlayerConfigurationFlag::sfxSoftAndHard);
}

void SfxExporter::encodeCellSharedHardwareOrSoftwareAndHardware(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters, const bool hardwareOnly,
                                                                PlayerConfiguration& playerConfiguration) noexcept
{
    const auto isNoise = channelRegisters.isNoise();
    jassert(channelRegisters.getVolume() == 16);
    const auto isRetrig = channelRegisters.isRetrig();
    const auto hardwareEnvelope = channelRegisters.getHardwareEnvelope();
    jassert((hardwareEnvelope >= 8) && (hardwareEnvelope <= 15));
    const auto encodedHardwareEnvelope = (static_cast<unsigned int>(hardwareEnvelope - 8) & 0b111U);
    BitNumber bitNumber;
    bitNumber.injectBits(hardwareOnly ? "10" : "11");
    bitNumber.injectBool(isRetrig);
    bitNumber.injectNumber(static_cast<int>(encodedHardwareEnvelope), 3U);
    bitNumber.injectBool(false);        // Ignored bit.
    bitNumber.injectBool(isNoise);
    const auto commentType = hardwareOnly ? juce::String("Hard only") : "Soft and Hard";
    auto comment = commentType + ". Env: " + juce::String(hardwareEnvelope) + ".";
    if (isRetrig) {
        comment += " Retrig.";
    }
    sourceGenerator.declareByte(bitNumber.get(), comment);

    encodeNoiseIfAny(sourceGenerator, channelRegisters);
    playerConfiguration.addFlag(isNoise, hardwareOnly ? PlayerConfigurationFlag::sfxHardOnlyNoise : PlayerConfigurationFlag::sfxSoftAndHardNoise);
    playerConfiguration.addFlag(isRetrig, hardwareOnly ? PlayerConfigurationFlag::sfxHardOnlyRetrig : PlayerConfigurationFlag::sfxSoftAndHardRetrig);

    // Encodes the hardware period.
    const auto hardwarePeriod = channelRegisters.getHardwarePeriod();
    sourceGenerator.declareWord(hardwarePeriod, "Hardware period.");
}

void SfxExporter::encodeNoiseIfAny(SourceGenerator& sourceGenerator, const ChannelOutputRegisters& channelRegisters) noexcept
{
    if (!channelRegisters.isNoise()) {
        return;
    }

    const auto noise = channelRegisters.getNoise();
    jassert((noise > 0) && (noise <= 31));
    sourceGenerator.declareByte(noise, "Noise: " + juce::String(noise) + ".");
}


// Sfx Data Provider method implementations.
// ==============================================

SfxExporter::SfxDataProvider::SfxDataProvider(const Song& pSong, PsgPart pPsgPart) noexcept :
        song(pSong),
        psgPart(std::move(pPsgPart))
{
}

OptionalId SfxExporter::SfxDataProvider::getInstrumentIdFromAudioThread(const int instrumentIndex) const noexcept
{
    return song.getInstrumentId(instrumentIndex);
}

InstrumentType SfxExporter::SfxDataProvider::getInstrumentTypeFromAudioThread(const OptionalId& /*instrumentId*/) const noexcept
{
    return InstrumentType::psgInstrument;
}

OptionalId SfxExporter::SfxDataProvider::getExpressionIdFromAudioThread(bool /*isArpeggio*/, int /*expressionIndex*/) const noexcept
{
    jassertfalse;
    return { };
}

SongDataProvider::ExpressionMetadata SfxExporter::SfxDataProvider::getExpressionMetadataFromAudioThread(bool /*isArpeggio*/, const OptionalId& /*expressionId*/) const noexcept
{
    return { 0, 0, 0 };
}

SongDataProvider::PsgInstrumentFrameData SfxExporter::SfxDataProvider::getPsgInstrumentFrameDataFromAudioThread(const OptionalId& /*instrumentId*/, const int cellIndex) const noexcept
{
    const auto cell = psgPart.buildLowLevelCell(cellIndex);
    const auto speed = psgPart.getSpeed();
    const auto loop = psgPart.getMainLoop();
    const auto isInstrumentRetrig = psgPart.isInstrumentRetrig();
    return { loop, speed, isInstrumentRetrig, cell };
}

SongDataProvider::SampleInstrumentFrameData SfxExporter::SfxDataProvider::getSampleInstrumentFrameDataFromAudioThread(const OptionalId& /*instrumentId*/) const noexcept
{
    jassertfalse;       // Should not be called.
    return { Loop(), 0, 0.0F, nullptr };
}

int SfxExporter::SfxDataProvider::getExpressionValueFromAudioThread(bool /*isArpeggio*/, const OptionalId& /*expressionId*/, int /*cellIndex*/) const noexcept
{
    return 0;
}

std::pair<int, float> SfxExporter::SfxDataProvider::getPsgFrequencyFromChannelFromAudioThread(int /*channelIndexInSong*/) const noexcept
{
    const auto psgs = song.getSubsongPsgs(song.getFirstSubsongId());
    jassert(!psgs.empty());
    const auto psg = psgs.at(0U);
    return { psg.getPsgFrequency(), psg.getReferenceFrequency() };
}

int SfxExporter::SfxDataProvider::getTranspositionFromAudioThread(int /*channelIndexInSong*/) const noexcept
{
    return 0;
}

bool SfxExporter::SfxDataProvider::isEffectContextEnabled() const noexcept
{
    return false;
}

LineContext SfxExporter::SfxDataProvider::determineEffectContextFromAudioThread(CellLocationInPosition /*location*/) const noexcept
{
    return { };
}

LineContext SfxExporter::SfxDataProvider::determineEffectContextFromAudioThread(int /*channelIndexInSong*/) const noexcept
{
    return { };
}

}   // namespace arkostracker
