#include "SampleExporter.h"

#include "SampleCodeGenerator.h"
#include "SampleEncoder.h"
#include "../../business/instrument/SampleResampler.h"
#include "../../business/song/tool/InstrumentCounter.h"
#include "../../song/Song.h"
#include "../sourceGenerator/SourceGenerator.h"

namespace arkostracker
{
SampleExporter::SampleExporter(const Song& pSong, ExportConfiguration pExportConfiguration) noexcept :
        song(pSong),
        exportConfiguration(pExportConfiguration)
{
}


// Task method implementations.
// ===============================

std::pair<bool, std::unique_ptr<SongExportResult>> SampleExporter::performTask() noexcept
{
    auto exportResult = std::make_unique<SongExportResult>();

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    const auto baseLabel = getSampleBaseLabel();

    sourceGenerator.setPrefixForDisark(baseLabel);
    const auto songName = song.getName();
    sourceGenerator.declareComment((songName.isEmpty() ? "" : (songName + ", ")) + "Samples, format v1.0.").addEmptyLine();
    sourceGenerator.declareComment("Generated by Arkos Tracker 3.").addEmptyLine();

    // Org?
    if (exportConfiguration.getAddress().isPresent()) {
        sourceGenerator.declareAddressChange(exportConfiguration.getAddress().getValue()).addEmptyLine();
    }

    sourceGenerator.declareLabel(baseLabel);
    sourceGenerator.declareExternalLabel(baseLabel).addEmptyLine();

    const auto instrumentCounterResult = InstrumentCounter::countInstruments(song);
    const auto sampleFlags = exportConfiguration.getSampleEncoderFlags();

    // Builds the index table.
    if (sampleFlags.isGenerateIndexTable()) {
        sourceGenerator.declareLabel(baseLabel + "TableIndex", false, "The samples table index, starting at 1.");
        sourceGenerator.declarePointerRegionStart();   // Note that we do not manage the "0" below differently for Disark. Who cares?

        for (const auto& entry: instrumentCounterResult) {
            const auto encodeSample = mustEncodeSample(entry);

            sourceGenerator.declareWord((encodeSample ? getSampleLabel(entry.getIndex()) : "0"),
                "Instrument index " + juce::String(entry.getIndex()));
        }

        sourceGenerator.declarePointerRegionEnd();
        sourceGenerator.addEmptyLine();
    }


    PlayerConfiguration playerConfiguration(false);

    // Encodes each Sample that needs to be.
    for (const auto& entry: instrumentCounterResult) {
        if (!mustEncodeSample(entry)) {
            continue;
        }
        const auto sampleIndex = entry.getIndex();

        sourceGenerator.declareComment("The data for sample index " + juce::String(sampleIndex) + ".");
        sourceGenerator.declareLabel(getSampleLabel(sampleIndex));

        song.performOnConstInstrumentFromIndex(sampleIndex, [&](const Instrument& originalInstrument) {
            // Important to apply the resampling first.
            const auto newSamplePart = SampleResampler::resample(originalInstrument.getConstSamplePart());

            SampleCodeGenerator::generateHeader(sourceGenerator, sampleFlags, newSamplePart);
            SampleCodeGenerator::generateSampleData(sourceGenerator, sampleFlags, newSamplePart);
        });

        sourceGenerator.addEmptyLine();
    }
    sourceGenerator.declareEndOfFile();

    auto result = std::make_unique<SongExportResult>(std::move(songOutputStream), playerConfiguration);
    return { true, std::move(result) };
}


// =====================================================

juce::String SampleExporter::getSampleBaseLabel(const bool addEndSeparator) const noexcept
{
    const auto endSeparator = addEndSeparator ? juce::String("_") : juce::String();
    return exportConfiguration.getBaseLabel() + "Sample" + endSeparator;}

juce::String SampleExporter::getSampleLabel(const int sampleIndex) const noexcept
{
    return getSampleBaseLabel(true) + "Sample" + juce::String(sampleIndex);

}

bool SampleExporter::mustEncodeSample(const InstrumentCounter::InstrumentResult& entry) const noexcept
{
    // Only samples are encoded. And only the ones that are used, unless we want all of them.
    return entry.isSample() && (entry.getCount() > 0 || !exportConfiguration.getSampleEncoderFlags().isIgnoreUnusedSamples());
}

}   // namespace arkostracker
