#include "RawLinearExport.h"

#include "../../business/song/tool/frameCounter/FrameCounter.h"
#include "../../player/SongPlayer.h"
#include "../../song/Song.h"
#include "../SongExportResult.h"
#include "../sourceGenerator/SourceGenerator.h"

namespace arkostracker
{

static constexpr auto dataTypeNote = 0;
static constexpr auto dataTypeWait = 1;
static constexpr auto dataTypeEnd = 255;

RawLinearExporter::RawLinearExporter(const std::shared_ptr<Song>& pSong, ExportConfiguration pExportConfiguration, std::set<InstrumentType> pInstrumentTypes) noexcept :
        song(pSong),
        exportConfiguration(std::move(pExportConfiguration)),
        instrumentTypes(std::move(pInstrumentTypes))
{
}

std::pair<bool, std::unique_ptr<SongExportResult>> RawLinearExporter::performTask() noexcept
{
    auto exportResult = std::make_unique<SongExportResult>();
    auto success = true;

    jassert(exportConfiguration.getSubsongIds().size() == 1);       // Only one subsong is supported.
    const auto subsongId = exportConfiguration.getFirstSubsongId();

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    const auto songName = song->getName();
    sourceGenerator.declareComment((songName.isEmpty() ? "" : (songName + ", ")) + "Linear RAW format V1.0.").addEmptyLine();
    sourceGenerator.declareComment("Generated by Arkos Tracker 3.").addEmptyLine();

    // Org?
    if (exportConfiguration.getAddress().isPresent()) {
        sourceGenerator.declareAddressChange(exportConfiguration.getAddress().getValue()).addEmptyLine();
    }

    const auto baseLabel = getBaseLabel();
    sourceGenerator.declareLabel(baseLabel);
    sourceGenerator.declareExternalLabel(baseLabel).addEmptyLine();

    generateData(subsongId, sourceGenerator);

    sourceGenerator.declareEndOfFile();

    exportResult->setSongStream(std::move(songOutputStream));

    return { success, std::move(exportResult) };
}

juce::String RawLinearExporter::getBaseLabel() const noexcept
{
    return exportConfiguration.getBaseLabel();
}

void RawLinearExporter::generateData(const Id& subsongId, SourceGenerator& sourceGenerator) const noexcept
{
    const auto loopLabel = getBaseLabel() + "loop";

    SongPlayer songPlayer(song);
    const auto startLocation = Location(subsongId, 0);
    const auto [_, newPastEndLocation] = song->getLoopStartAndPastEndPositions(subsongId);
    const auto psgCount = song->getPsgCount(subsongId);

    const auto[counter, upTo, loopToCounter] = FrameCounter::count(*song, subsongId);
    const auto iterationCount = counter;
    jassert(iterationCount > 0);

    songPlayer.play(startLocation, startLocation, newPastEndLocation, true, true);

    auto waitCounter = 0;

    for (auto iterationIndex = 0; iterationIndex < iterationCount; ++iterationIndex) {
        for (auto browsedPsgIndex = 0; browsedPsgIndex < psgCount; ++browsedPsgIndex) {
            // Necessary to get all the registers of all the PSGs.
            auto [psgRegistersPtr, __] = songPlayer.getNextRegisters(browsedPsgIndex);

            // Loop to encode?
            if (iterationIndex == loopToCounter) {
                // Before encoding the loop label, encodes the possible pending wait.
                encodeWait(sourceGenerator, waitCounter);
                sourceGenerator.declareLabel(loopLabel);
            }

            const auto result = songPlayer.getResultsOffline();
            jassert(result.size() == 3);

            // Browses each channel.
            auto channelIndex = 0;
            for (const auto& channelResult : result) {
                const auto noteOptional = channelResult->getNewPlayedNote();
                const auto instrumentIdOptional = channelResult->getInstrumentId();
                if (noteOptional.isPresent() && instrumentIdOptional.isPresent()) {
                    // Is the instrument type among the requested ones?
                    const auto instrumentId = instrumentIdOptional.getValue();
                    const auto instrumentIndexOptional = song->getInstrumentIndex(instrumentIdOptional.getValue());
                    if (instrumentIndexOptional.isAbsent()) {
                        jassertfalse;       // Abnormal, should be present.
                    } else {
                        auto found = false;
                        const auto instrumentIndex = instrumentIndexOptional.getValue();
                        song->performOnConstInstrument(instrumentId, [&](const Instrument& instrument) {
                            if (instrumentTypes.find(instrument.getType()) != instrumentTypes.cend()) {
                                found = true;
                            }
                        });

                        if (found) {
                            // Before encoding the code, encodes the possible wait.
                            encodeWait(sourceGenerator, waitCounter);

                            // Encodes the note.
                            sourceGenerator.declareByte(dataTypeNote, "Note command.");
                            sourceGenerator.declareByte(noteOptional.getValue(), "Note.");
                            sourceGenerator.declareByte(instrumentIndex, "Instrument index.");
                            sourceGenerator.declareByte(channelIndex, "Channel index.");
                            sourceGenerator.addEmptyLine();
                        }
                    }
                }

                ++channelIndex;
            }

            ++waitCounter;
        }
    }

    // Before encoding the end, makes sure there are no pending wait.
    encodeWait(sourceGenerator, waitCounter);

    // Encodes the end.
    sourceGenerator.declareByte(dataTypeEnd, "End command.");
    sourceGenerator.declareWord(loopLabel, "Loop here.");
    sourceGenerator.addEmptyLine();
}

void RawLinearExporter::encodeWait(SourceGenerator& sourceGenerator, int& waitCounter) noexcept
{
    if (waitCounter <= 0) {
        return;
    }

    sourceGenerator.declareByte(dataTypeWait, "Wait command.");
    sourceGenerator.declareWord(waitCounter);
    sourceGenerator.addEmptyLine();

    waitCounter = 0;
}

}   // namespace arkostracker
