#include "RawExporter.h"

#include <algorithm>

#include "../../business/instrument/SampleResampler.h"
#include "../../business/song/tool/optimizers/SongOptimizer.h"
#include "../../export/samples/SampleCodeGenerator.h"
#include "../../song/Song.h"
#include "../../utils/BitNumber.h"
#include "../../utils/NumberUtil.h"
#include "../../utils/PsgValues.h"

namespace arkostracker
{

RawExporter::RawExporter(const Song& pSong, const EncodedDataFlag& pEncodedDataFlag, ExportConfiguration pExportConfiguration) noexcept :
        originalSong(pSong),
        song(),
        originalSubsongId(pExportConfiguration.getFirstSubsongId()),
        subsongId(),
        encodedDataFlag(pEncodedDataFlag),
        exportConfiguration(std::move(pExportConfiguration)),
        subsongIndex(),
        maximumHeight(0),
        trackCount(),
        speedTrackCount(),
        eventTrackCount()
{
    jassert(exportConfiguration.getSubsongIds().size() == 1U);
}


// Task method implementations.
// ======================================================

std::pair<bool, std::unique_ptr<SongExportResult>> RawExporter::performTask() noexcept
{
    // Optimize, keep only the selected Subsong.
    // Also, do NOT convert Legato to effect.
    song = SongOptimizer::optimize(originalSong, true, true,
        true, exportConfiguration.areSamplesExported(), true, true, true,
        true, true, false, { originalSubsongId });
    if (song == nullptr) {
        jassertfalse;       // Shouldn't happen.
        return { false, nullptr };
    }
    subsongId = song->getFirstSubsongId();  // New song, so the Subsong ID has changed!

    auto exportResult = std::make_unique<SongExportResult>();

    const auto subsongIndexOptional = song->getSubsongIndex(subsongId);
    if (subsongIndexOptional.isAbsent()) {
        jassertfalse;       // Shouldn't happen!
        return { false, nullptr };
    }
    subsongIndex = subsongIndexOptional.getValue();

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    const auto baseLabel = getBaseLabel(false);

    //sourceGenerator.setPrefixForDisark(baseLabel);
    const auto songName = song->getName();
    sourceGenerator.declareComment((songName.isEmpty() ? "" : (songName + ", ")) + "RAW format V1.1.").addEmptyLine();
    sourceGenerator.declareComment("Generated by Arkos Tracker 3.").addEmptyLine();

    // Org?
    if (exportConfiguration.getAddress().isPresent()) {
        sourceGenerator.declareAddressChange(exportConfiguration.getAddress().getValue()).addEmptyLine();
    }

    sourceGenerator.declareLabel(baseLabel);
    sourceGenerator.declareExternalLabel(baseLabel).addEmptyLine();

    sourceGenerator.declareComment("Song: " + song->getName() + ", Subsong: " + song->getSubsongMetadata(subsongId).getName());

    // Sets the track count.
    song->performOnConstSubsong(subsongId, [&](const Subsong& subsong) {
        trackCount = subsong.getTrackCount();
        speedTrackCount = subsong.getSpecialTrackCount(true);
        eventTrackCount = subsong.getSpecialTrackCount(false);
    });

    // Encodes each part of the file.
    encodeHeader(sourceGenerator);
    encodeLinker(sourceGenerator);
    // Encodes the normal/Speed/Event Track reference tables.
    if (encodedDataFlag.mustEncodeReferenceTables()) {
        encodeIndexTable(sourceGenerator, "Track index table", getTrackIndexesLabel(), trackCount,
                         [&](const int index) {
                             return getTrackLabel(index);
                         });
        if (encodedDataFlag.mustEncodeSpeedTrack()) {
            encodeIndexTable(sourceGenerator, "SpeedTrack index table", getSpeedTrackIndexesLabel(), speedTrackCount,
                             [&](const int index) {
                                 return getSpeedTrackLabel(index);
                             });
        }
        if (encodedDataFlag.mustEncodeEventTrack()) {
            encodeIndexTable(sourceGenerator, "EventTrack index table", getEventTrackIndexesLabel(), eventTrackCount,
                             [&](const int index) {
                                 return getEventTrackLabel(index);
                             });
        }
    }
    encodeTracks(sourceGenerator);
    if (encodedDataFlag.mustEncodeSpeedTrack()) {
        encodeSpecialTracks(sourceGenerator, true);
    }
    if (encodedDataFlag.mustEncodeEventTrack()) {
        encodeSpecialTracks(sourceGenerator, false);
    }

    auto success = true;

    // Encodes the Arpeggios, and also the index table.
    if (encodedDataFlag.mustEncodeArpeggioTables()) {
        const auto arpeggioCount = song->getConstExpressionHandler(true).getCount();
        encodeIndexTable(sourceGenerator, "Arpeggio index table", getArpeggioIndexesLabel(), arpeggioCount,
                         [&](const int index) {
                             return getArpeggioLabel(index);
                         });
        success = success && encodeExpressions(sourceGenerator, true);
    }

    // Encodes the Pitches, and also the index table.
    if (encodedDataFlag.mustEncodePitchTables()) {
        const auto pitchCount = song->getConstExpressionHandler(false).getCount();
        encodeIndexTable(sourceGenerator, "Pitch index table", getPitchIndexesLabel(), pitchCount,
                         [&](const int index) {
                             return getPitchLabel(index);
                         });
        success = success && encodeExpressions(sourceGenerator, false);
    }

    // Encodes the Instruments and also the index table.
    if (encodedDataFlag.mustEncodeInstruments()) {
        const auto instrumentCount = song->getInstrumentCount();
        encodeIndexTable(sourceGenerator, "Instrument index table", getInstrumentIndexesLabel(), instrumentCount,
                         [&](const int index) {
                             return getInstrumentLabel(index);
                         });
        success = success && encodeInstruments(sourceGenerator);
        if (!success) {
            // The samples are probably too big.
            return { false, nullptr };
        }
    }

    sourceGenerator.declareEndOfFile();

    exportResult->setSongStream(std::move(songOutputStream));

    return { success, std::move(exportResult) };
}

juce::String RawExporter::getBaseLabel(const bool showSubsongIndex) const noexcept
{
    auto baseLabel = exportConfiguration.getBaseLabel();
    if (!showSubsongIndex) {
        return baseLabel;
    }
    return baseLabel + "Subsong" + juce::String(subsongIndex) + "_";
}

juce::String RawExporter::getLinkerLabel() const noexcept
{
    return getBaseLabel() + "Linker";
}
juce::String RawExporter::getLinkerLoopLabel() const noexcept
{
    return getBaseLabel() + "Loop";
}

juce::String RawExporter::getTrackIndexesLabel() const noexcept
{
    return getBaseLabel() + "TrackIndexes";
}
juce::String RawExporter::getSpeedTrackIndexesLabel() const noexcept
{
    return getBaseLabel() + "SpeedTrackIndexes";
}
juce::String RawExporter::getEventTrackIndexesLabel() const noexcept
{
    return getBaseLabel() + "EventTrackIndexes";
}

juce::String RawExporter::getInstrumentIndexesLabel() const noexcept
{
    return getBaseLabel(false) + "InstrumentIndexes";
}

juce::String RawExporter::getArpeggioIndexesLabel() const noexcept
{
    return getBaseLabel(false) + "ArpeggioIndexes";
}
juce::String RawExporter::getPitchIndexesLabel() const noexcept
{
    return getBaseLabel(false) + "PitchIndexes";
}

juce::String RawExporter::getInstrumentLabel(const int index) const noexcept
{
    return getBaseLabel(false) + "Instrument" + juce::String(index);
}

juce::String RawExporter::getTrackLabel(const int index) const noexcept
{
    return getBaseLabel() + "Track" + juce::String(index);
}
juce::String RawExporter::getSpeedTrackLabel(const int index) const noexcept
{
    return getSpecialTrackLabel(index, true);
}
juce::String RawExporter::getEventTrackLabel(const int index) const noexcept
{
    return getSpecialTrackLabel(index, false);
}
juce::String RawExporter::getSpecialTrackLabel(const int index, const bool isSpeedTrack) const noexcept
{
    return getBaseLabel()
        + (isSpeedTrack ? "SpeedTrack" : "EventTrack")
        + juce::String(index);
}

juce::String RawExporter::getArpeggioLabel(const int index) const noexcept
{
    return getBaseLabel() + "Arpeggio" + juce::String(index);
}
juce::String RawExporter::getPitchLabel(const int index) const noexcept
{
    return getBaseLabel() + "Pitch" + juce::String(index);
}

void RawExporter::encodeHeader(SourceGenerator& sourceGenerator) const noexcept
{
    sourceGenerator.declareComment("Header");

    // Builds and encodes the flag bytes.
    {
        BitNumber number;
        number.injectBool(encodedDataFlag.mustEncodeSongAndSubsongMetadata());
        number.injectBool(encodedDataFlag.mustEncodeReferenceTables());
        number.injectBool(encodedDataFlag.mustEncodeSpeedTrack());
        number.injectBool(encodedDataFlag.mustEncodeEventTrack());
        number.injectBool(encodedDataFlag.mustEncodeInstruments());
        number.injectBool(encodedDataFlag.mustEncodeArpeggioTables());
        number.injectBool(encodedDataFlag.mustEncodePitchTables());
        number.injectBool(encodedDataFlag.mustEncodeEffects());
        sourceGenerator.declareByte(number.get(), "Flag byte 1.");
    }
    {
        BitNumber number;
        number.injectBool(encodedDataFlag.mustEncodeRleForEmptyLines());
        number.injectBool(encodedDataFlag.mustEncodeTranspositions());
        number.injectBool(encodedDataFlag.mustEncodeHeight());
        sourceGenerator.declareByte(number.get(), "Flag byte 2.");
    }

    // Encodes Song/Subsong metadata, if wanted.
    if (encodedDataFlag.mustEncodeSongAndSubsongMetadata()) {
        const auto subsongMetadata = song->getSubsongMetadata(subsongId);
        const auto psgCount = song->getPsgCount(subsongId);

        sourceGenerator.declareComment("Song/Subsong metadata");

        sourceGenerator.declareZeroTerminatedString(song->getName(), "Song name.");
        sourceGenerator.declareZeroTerminatedString(song->getAuthor(), "Author.");
        sourceGenerator.declareZeroTerminatedString(song->getComposer(), "Composer.");
        sourceGenerator.declareZeroTerminatedString(song->getComments(), "Comments.");

        sourceGenerator.declareZeroTerminatedString(subsongMetadata.getName(), "Subsong title.").addEmptyLine();
        sourceGenerator.declareByte(subsongMetadata.getInitialSpeed(), "Initial speed.");
        sourceGenerator.declareByte(subsongMetadata.getDigiChannel(), "Digichannel.");
        sourceGenerator.declareByte(static_cast<int>(subsongMetadata.getReplayFrequencyHz()), "Replay rate, in Hz.");

        sourceGenerator.declareByte(PsgValues::getChannelCount(psgCount), "Channel count.");

        auto subsongLength = 0;
        auto loopStartPosition = 0;
        song->performOnConstSubsong(subsongId, [&](const Subsong& subsong) {
            subsongLength = subsong.getLength();
            loopStartPosition = subsong.getLoopStartPosition();
        });

        jassert(psgCount > 0);
        sourceGenerator.declareByte(psgCount, "Psg count.");
        sourceGenerator.declareByte(subsongLength, "Length in position (>0).");
        sourceGenerator.declareByte(loopStartPosition, "Loop To position index (>=0).");

        // Declares each PSG.
        const auto psgs = song->getSubsongPsgs(subsongId);
        for (auto psgIndex = 0U; psgIndex < psgs.size(); ++psgIndex) {
            const auto& psg = psgs.at(psgIndex);
            sourceGenerator.declareComment("Psg " + juce::String(psgIndex + 1));
            sourceGenerator.declareFourBytes(psg.getPsgFrequency(), "PSG frequency, in Hz.");
            sourceGenerator.declareWord(static_cast<int>(psg.getReferenceFrequency()), "Reference frequency, in Hz.");
            sourceGenerator.declareWord(psg.getSamplePlayerFrequency(), "Sample player frequency, in Hz.");
        }
    }
    sourceGenerator.addEmptyLine();

    // Encodes the reference tables indexes, if wanted.
    if (encodedDataFlag.mustEncodeReferenceTables()) {
        sourceGenerator.declareComment("Reference tables");

        sourceGenerator.declareWord(getLinkerLabel());
        sourceGenerator.declareWord(getTrackIndexesLabel());
        sourceGenerator.declareWord(encodedDataFlag.mustEncodeSpeedTrack() ? getSpeedTrackIndexesLabel() : "0", "Speed Track indexes.");
        sourceGenerator.declareWord(encodedDataFlag.mustEncodeEventTrack() ? getEventTrackIndexesLabel() : "0", "Event Track indexes.");
        sourceGenerator.declareWord(encodedDataFlag.mustEncodeInstruments() ? getInstrumentIndexesLabel() : "0", "Instrument indexes.");
        sourceGenerator.declareWord(encodedDataFlag.mustEncodeArpeggioTables() ? getArpeggioIndexesLabel() : "0", "Arpeggio indexes.");
        sourceGenerator.declareWord(encodedDataFlag.mustEncodePitchTables() ? getPitchIndexesLabel() : "0", "Pitch indexes.");
        sourceGenerator.addEmptyLine();
    }
}

void RawExporter::encodeLinker(SourceGenerator& sourceGenerator) noexcept
{
    // Encodes the Linker.
    sourceGenerator.declareComment("Linker");
    sourceGenerator.declareLabel(getLinkerLabel());

    song->performOnConstSubsong(subsongId, [&](const Subsong& subsong) {
        // Browses each Position.
        const auto [positionLoopStart, endPosition] = subsong.getLoopStartAndEndPosition();

        for (auto positionIndex = 0; positionIndex <= endPosition; ++positionIndex) {
            // Encodes the Loop label?
            if (positionIndex == positionLoopStart) {
                sourceGenerator.declareLabel(getLinkerLoopLabel());
            }

            const auto& position = subsong.getPositionRef(positionIndex);
            const auto pattern = subsong.getPatternFromIndex(position.getPatternIndex());
            const auto channelCount = pattern.getChannelCount();

            // New maximum height?
            maximumHeight = std::max(maximumHeight, position.getHeight());

            sourceGenerator.declareComment("Position " + juce::String(positionIndex) + ".");
            // Retrieves the Track indexes the user is interested in.
            for (auto channelIndex = 0; channelIndex < channelCount; ++channelIndex) {
                const auto trackIndex = pattern.getCurrentTrackIndex(channelIndex);
                sourceGenerator.declareWord(getTrackLabel(trackIndex));
            }
            sourceGenerator.addEmptyLine();

            // Encodes the SpeedTrack?
            if (encodedDataFlag.mustEncodeSpeedTrack()) {
                const auto speedTrack = pattern.getCurrentSpecialTrackIndex(true);
                sourceGenerator.declareWord(getSpecialTrackLabel(speedTrack, true), "Speed Track.");
            }

            // Encodes the EventTrack?
            if (encodedDataFlag.mustEncodeEventTrack()) {
                const auto eventTrack = pattern.getCurrentSpecialTrackIndex(false);
                sourceGenerator.declareWord(getSpecialTrackLabel(eventTrack, false), "Event Track.");
            }

            // Encodes the height?
            if (encodedDataFlag.mustEncodeHeight()) {
                sourceGenerator.declareByte(position.getHeight(), "Height (>0).");
            }

            // Encode the transpositions?
            if (encodedDataFlag.mustEncodeEventTrack()) {
                for (auto channelIndex = 0; channelIndex < channelCount; ++channelIndex) {
                    const auto transposition = position.getTransposition(channelIndex);
                    sourceGenerator.declareByte(transposition);
                }
                sourceGenerator.declareComment("Transpositions.");
            }

            sourceGenerator.addEmptyLine();
        }
    });

    // Encodes the end of the Linker, with the loop.
    sourceGenerator.declareWord(0, "End of Linker, with the loop.").declareWord(getLinkerLoopLabel());

    sourceGenerator.addEmptyLine();
}

void RawExporter::encodeTracks(SourceGenerator& sourceGenerator) const noexcept
{
    jassert(maximumHeight > 0);

    sourceGenerator.declareComment("Tracks");

    // Encodes each Track that is used.
    for (auto trackIndex = 0; trackIndex < trackCount; ++trackIndex) {
        sourceGenerator.declareLabel(getTrackLabel(trackIndex));

        Track track;
        song->performOnConstSubsong(subsongId, [&](const Subsong& subsong) {
            track = subsong.getTrackRefFromIndex(trackIndex);
        });

        // Browses through the maximum height, so that all the Tracks may be the same size.
        auto consecutiveEmptyCells = 0;
        for (auto cellIndex = 0; cellIndex < maximumHeight; ++cellIndex) {
            if (const auto& cell = track.getCellRefConst(cellIndex); cell.isEmpty()) {
                ++consecutiveEmptyCells;
            } else {
                // Encodes the possible empty cells.
                encodeEmptyCells(sourceGenerator, consecutiveEmptyCells);
                consecutiveEmptyCells = 0;

                // What to encode?
                const auto isNotePresent = cell.isNote();
                auto instrumentNumberToEncode = cell.isInstrument() ? cell.getInstrument().getValue() : 0;  // By default, if no note, we don't care about the Instrument.

                const auto noteToEncode = isNotePresent ? cell.getNote().getValue().getNote() : noteValueNoNoteButEffects;      // No note? The encoded note is a special one indicating effects are present.
                const auto legato = cell.isLegato();

                // Encodes the note and instrument.
                sourceGenerator.declareByte(noteToEncode, isNotePresent ? "Note." : "No note.");
                auto instrumentComment = isNotePresent ? juce::String("Instrument.") : "No instrument.";
                if (legato) {
                    instrumentNumberToEncode = 255;
                    instrumentComment += " Legato.";
                }
                sourceGenerator.declareByte(instrumentNumberToEncode, instrumentComment);

                // Encodes the effects, if wanted (only those that are not empty).
                if (encodedDataFlag.mustEncodeEffects()) {
                    const auto& cellEffects = cell.getEffects();
                    for (const auto& cellEffect : cellEffects.getPresentEffects()) {
                        const auto effectNumber = static_cast<int>(cellEffect.getEffect());
                        auto effectValue = cellEffect.getEffectRawValue();

                        // If the effect is a Pitch, multiplies the value by the Pitch Track Effect Ratio (checking the limits).
                        // This is especially useful when using the Pitch for a MOD player on hardware: the way the periods are managed can be
                        // quite different from the C++ player!
                        if (EffectUtil::isPitchEffect(effectNumber)) {
                            effectValue = static_cast<int>(
                                static_cast<double>(NumberUtil::correctNumber(effectValue, 0, 0xffff)) * encodedDataFlag.getPitchTrackEffectRatio()
                            );
                        }

                        // Encodes the effect and value.
                        encodeEffect(sourceGenerator, effectNumber, effectValue);
                    }

                    // Encodes the effect end.
                    sourceGenerator.declareByte(0, "End of effects.");
                }

                sourceGenerator.addEmptyLine();
            }
        }

        // Encodes the possible empty cells at the end of the Track.
        encodeEmptyCells(sourceGenerator, consecutiveEmptyCells);
    }
}

void RawExporter::encodeEmptyCells(SourceGenerator& sourceGenerator, const int consecutiveEmptyCells) const noexcept
{
    jassert((consecutiveEmptyCells >= 0) && (consecutiveEmptyCells < 128));

    if (consecutiveEmptyCells > 0) {
        // Encodes as RLE?
        if (encodedDataFlag.mustEncodeRleForEmptyLines()) {
            // RLE.
            sourceGenerator.declareByte(127 + consecutiveEmptyCells, juce::String(consecutiveEmptyCells) + " empty cells.");
        } else {
            // Not RLE.
            for (auto i = 0; i < consecutiveEmptyCells; ++i) {
                sourceGenerator.declareByte(128, "One empty cell.");
            }
        }
        sourceGenerator.addEmptyLine();
    }
}

void RawExporter::encodeEffect(SourceGenerator& sourceGenerator, const int effectNumber, const int effectValue) noexcept
{
    jassert(effectNumber < 256);
    jassert(effectValue <= 0xffff);

    // Encodes the effect and value.
    sourceGenerator.declareByte(effectNumber, "Effect number: " + juce::String(effectNumber));
    sourceGenerator.declareWord(effectValue, "Effect value: " + juce::String(effectValue));
}

void RawExporter::encodeSpecialTracks(SourceGenerator& sourceGenerator, const bool isSpeedTrack) const noexcept
{
    // Since Speed and Events Tracks are coded the exact same way, I use this boolean trick to share the code.

    jassert(maximumHeight > 0);

    sourceGenerator.declareComment(isSpeedTrack ? "SpeedTracks" : "EventTracks");

    // Encodes each Special Track.

    // Encodes each Track that is used.
    const auto specialTrackCount = isSpeedTrack ? speedTrackCount : eventTrackCount;
    for (auto specialTrackIndex = 0; specialTrackIndex < specialTrackCount; ++specialTrackIndex) {
        sourceGenerator.declareLabel(getSpecialTrackLabel(specialTrackIndex, isSpeedTrack));

        SpecialTrack specialTrack;
        song->performOnConstSubsong(subsongId, [&](const Subsong& subsong) {
            specialTrack = subsong.getSpecialTrackRefFromIndex(specialTrackIndex, isSpeedTrack);
        });

        // Browses through the maximum height, so that all the Tracks may be the same size.
        for (auto cellIndex = 0; cellIndex < maximumHeight; ++cellIndex) {
            const auto specialCell = specialTrack.getCell(cellIndex);
            sourceGenerator.declareByte(specialCell.getValue());
        }

        sourceGenerator.addEmptyLine();
    }
}

bool RawExporter::encodeExpressions(SourceGenerator& sourceGenerator, const bool isArpeggio) const noexcept
{
    auto success = true;
    sourceGenerator.declareComment(isArpeggio ? "Arpeggios" : "Pitches");

    const auto& expressionHandler = song->getConstExpressionHandler(isArpeggio);
    const auto expressionCount = expressionHandler.getCount();

    // Browses the Expression.
    for (auto expressionIndex = 0; expressionIndex < expressionCount; ++expressionIndex) {
        sourceGenerator.declareLabel(isArpeggio ? getArpeggioLabel(expressionIndex) : getPitchLabel(expressionIndex));
        const auto expressionIdOptional = expressionHandler.find(expressionIndex);
        if (expressionIdOptional.isAbsent()) {
            success = false;
            jassertfalse;       // Not supposed to happen.
            break;
        }

        const auto expression = expressionHandler.getExpressionCopy(expressionIdOptional.getValue());
        encodeExpression(sourceGenerator, expression, isArpeggio);
    }

    sourceGenerator.addEmptyLine();

    return success;
}

void RawExporter::encodeExpression(SourceGenerator& sourceGenerator, const Expression& expression, const bool isArpeggio) noexcept
{
    // Encodes the header.
    sourceGenerator.declareByte(expression.getLength(), "Length.");
    const auto endIndex = expression.getEnd();
    const auto loopIndex = expression.getLoopStart();
    sourceGenerator.declareByte(endIndex, "End index.");
    sourceGenerator.declareByte(loopIndex, "Start loop index.");
    sourceGenerator.declareByte(expression.getSpeed(), "Speed (>=0).");

    // Encodes each values.
    for (auto index = 0; index <= endIndex; ++index) {
        const auto value = expression.getValue(index);
        if (isArpeggio) {
            sourceGenerator.declareByte(value);
        } else {
            sourceGenerator.declareWord(value);
        }
    }
}

void RawExporter::encodeIndexTable(SourceGenerator& sourceGenerator, const juce::String& firstComment, const juce::String& firstLabel,
                                   const int count, const std::function<juce::String(int)>& indexToLabelFunction) noexcept
{
    sourceGenerator.declareComment(firstComment);
    sourceGenerator.declareLabel(firstLabel);

    for (auto index = 0; index < count; ++index) {
        sourceGenerator.declareWord(indexToLabelFunction(index));
    }
    // Marks the end with a #ffff.
    sourceGenerator.declareWord(0xffff, "End marker.");

    sourceGenerator.addEmptyLine();
}

bool RawExporter::encodeInstruments(SourceGenerator& sourceGenerator) const noexcept
{
    sourceGenerator.declareComment("Instruments");

    auto success = true;
    auto instrumentIndex = 0;
    song->performOnConstInstruments([&] (const std::vector<std::unique_ptr<Instrument>>& instruments) {
        for (const auto& instrument : instruments) {
            if (const auto instrumentType = instrument->getType(); instrumentType == InstrumentType::psgInstrument) {
                encodePsgInstrument(sourceGenerator, instrumentIndex, *instrument);
            } else if (instrumentType == InstrumentType::sampleInstrument) {
                // A problem with the sample size?
                success = success && encodeSampleInstrument(sourceGenerator, instrumentIndex, *instrument);
            }
            ++instrumentIndex;
        }
    });

    sourceGenerator.addEmptyLine();

    return success;
}

void RawExporter::encodePsgInstrument(SourceGenerator& sourceGenerator, const int instrumentIndex, const Instrument& instrument) const noexcept
{
    sourceGenerator.declareLabel(getInstrumentLabel(instrumentIndex));

    const auto& psgPart = instrument.getConstPsgPart();

    // Encodes the header.
    sourceGenerator.declareByte(0).declareComment("Instrument type (PSG).");
    sourceGenerator.declareByte(psgPart.getLength(), "Length.");
    const auto mainLoop = psgPart.getMainLoop();
    const auto endIndex = mainLoop.getEndIndex();
    const auto loopStartIndex = mainLoop.getStartIndex();
    const auto loop = mainLoop.isLooping();
    sourceGenerator.declareByte(endIndex, "End index.");
    sourceGenerator.declareByte(loopStartIndex, "Loop start index.");
    sourceGenerator.declareByte(NumberUtil::boolToInt(loop), "Loop?");
    sourceGenerator.declareByte(psgPart.getSpeed(), "Speed.");
    sourceGenerator.declareByte(NumberUtil::boolToInt(psgPart.isInstrumentRetrig()), "Retrig?");
    sourceGenerator.addEmptyLine();

    // Encodes the data.
    for (auto cellIndex = 0; cellIndex <= endIndex; ++cellIndex) {
        // Encodes the Cell.
        sourceGenerator.declareComment("Cell " + juce::String(cellIndex));

        const auto cell = psgPart.buildLowLevelCell(cellIndex);
        const auto link = cell.getLink();
        sourceGenerator.declareByte(static_cast<int>(cell.getLink()), "Link: " + PsgInstrumentCellLinkHelper::toString(link) + ".");
        sourceGenerator.declareByte(cell.getVolume(), "Volume (0-15).");
        sourceGenerator.declareByte(cell.getNoise(), "Noise (0-31).");
        sourceGenerator.declareWord(cell.getSoftwarePeriod(), "Software period (0 for auto).");
        sourceGenerator.declareWord(cell.getSoftwarePitch(), "Software pitch.");
        sourceGenerator.declareByte(cell.getSoftwareArpeggio(), "Software arpeggio.");
        sourceGenerator.declareByte(cell.getRatio(), "Ratio (0-7).");
        sourceGenerator.declareByte(cell.getHardwareEnvelope(), "Hardware envelope (8-15).");
        sourceGenerator.declareWord(cell.getHardwarePeriod(), "Hardware period (0 for auto).");
        sourceGenerator.declareWord(cell.getHardwarePitch(), "Hardware pitch.");
        sourceGenerator.declareByte(cell.getHardwareArpeggio(), "Hardware arpeggio.");
        sourceGenerator.declareByte(NumberUtil::boolToInt(cell.isRetrig()), "Retrig?");

        sourceGenerator.addEmptyLine();
    }
}

bool RawExporter::encodeSampleInstrument(SourceGenerator& sourceGenerator, const int instrumentIndex, const Instrument& instrument) const noexcept
{
    jassert(instrument.getType() == InstrumentType::sampleInstrument);
    // Important to resample the sample, it is not done automatically.
    const auto& samplePart = SampleResampler::resample(instrument.getConstSamplePart());

    sourceGenerator.declareLabel(getInstrumentLabel(instrumentIndex));
    const auto sampleEncoderFlags = exportConfiguration.getSampleEncoderFlags();

    // Encodes the header.
    sourceGenerator.declareByte(1, "Instrument type (Sample).");
    const auto success = SampleCodeGenerator::generateHeader(sourceGenerator, sampleEncoderFlags, samplePart);
    // Security about the size of the sample.
    if (!success) {
        return false;
    }

    // Encodes the Sample itself.
    SampleCodeGenerator::generateSampleData(sourceGenerator, sampleEncoderFlags, samplePart);

    sourceGenerator.addEmptyLine();

    return true;
}

}   // namespace arkostracker
