#include "FapExporter.h"

#include "../../../thirdParty/fap/FapCrunch.h"
#include "../../utils/FileExtensions.h"
#include "../../utils/FileUtil.h"
#include "../../utils/MemoryBlockUtil.h"
#include "../ym/YmExporter.h"

namespace arkostracker
{
FapExporter::FapExporter(const std::shared_ptr<Song>& pSong, const Id& pSubsongId) noexcept :
        song(pSong),
        subsongId(pSubsongId),
        outputStream()
{
}

std::pair<bool, std::unique_ptr<FapResult>> FapExporter::performTask() noexcept
{
    // First, converts to YM. Must be interleaved, slower, but FAP requires that.
    ymExporter = std::make_unique<YmExporter>(song, subsongId, 0, true);
    ymExporter->setProgressListener(this);  // Needed because inner task.
    const auto [ymSuccess, ymMemoryOutputStream] = ymExporter->performTask();

    if (!ymSuccess) {
        jassertfalse;
        return { false, nullptr };
    }

    // Creates a temp file with the content.
    const auto inputFile = juce::File::createTempFile(FileExtensions::ymExtensionWithoutDot);
    const auto inputFileFullPath = inputFile.getFullPathName();
    const auto ymMemoryBlock = ymMemoryOutputStream->getMemoryBlock();
    auto success = FileUtil::saveMemoryBlockToFile(inputFileFullPath, ymMemoryBlock);
    if (!success) {
        jassertfalse;
        return { false, nullptr };
    }

    const auto tempFolder = juce::File::getSpecialLocation(juce::File::tempDirectory);
    const auto outputFile = juce::File(tempFolder.getFullPathName() + tempFolder.getSeparatorChar() + "fapTempAt3.fap");

    const std::vector arguments = {
        inputFile.getFullPathName().toStdString(),
        outputFile.getFullPathName().toStdString(),
    };

    // Then, converts to FAP.
    // With thanks from https://stackoverflow.com/a/39883532.
    std::vector<char*> argv;
    argv.push_back(nullptr);        // A dummy value for the program name.
    for (const auto& arg : arguments) {
        argv.push_back(const_cast<char*>(arg.data()));
    }
    argv.push_back(nullptr);        // End of the list.

    const auto fapResult = fapCrunch(static_cast<int>(argv.size() - 1), argv.data());
    success = (fapResult == 0);
    jassert(success);

    const auto inputFileStream = std::make_unique<juce::FileInputStream>(outputFile);
    success = success && inputFileStream->openedOk();
    jassert(success);

    auto generatedOutputMemoryBlock = MemoryBlockUtil::fromInputStream(*inputFileStream);

    // Clean.
    (void)inputFile.deleteFile();
    (void)outputFile.deleteFile();

    auto result = std::make_unique<FapResult>(
        generatedOutputMemoryBlock, getBufferSize(), getPlayTimeInNops(), getRegisterCountToPlay(), isR12Constant()     // Duplicates the buffer. Who cares.
    );

    return { success, std::move(result) };
}

void FapExporter::onAskedToCancelTask() noexcept
{
    ymExporter->askToCancelTask();
}

}   // namespace arkostracker
