#include "EventsExporter.h"

#include "../../business/song/tool/frameCounter/FrameCounter.h"

namespace arkostracker
{

EventsExporter::EventsExporter(std::shared_ptr<const Song> pSong, std::set<Type> pTypesToExport, ExportConfiguration pExportConfiguration) noexcept :
        song(std::move(pSong)),
        typesToExport(std::move(pTypesToExport)),
        subsongId(pExportConfiguration.getFirstSubsongId()),
        exportConfiguration(std::move(pExportConfiguration)),
        sampleIndexes()
{
    jassert(exportConfiguration.getSubsongIds().size() == 1U);
    jassert(!typesToExport.empty());        // Must not be empty, else nothing is going to be exported!
}

juce::String EventsExporter::getEventsLabel() const noexcept
{
    return exportConfiguration.getBaseLabel() + "Events";
}

juce::String EventsExporter::getLoopLabel() const noexcept
{
    return getEventsLabel() + "_Loop";
}

std::pair<bool, std::unique_ptr<SongExportResult>> EventsExporter::performTask() noexcept
{
    fillSampleIndexes();

    auto exportResult = std::make_unique<SongExportResult>();

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    sourceGenerator.setPrefixForDisark(exportConfiguration.getBaseLabel());

    encodeHeader(sourceGenerator);

    sourceGenerator.addEmptyLine().declareByteRegionStart();       // One big region for the whole.

    // Counts how many iterations are in the Song.
    const auto[iterationCount, upTo, loopToCounter] = FrameCounter::count(*song, subsongId);

    // Re-initializes the player.
    SongPlayer songPlayer(song);
    const auto startLocation = Location(subsongId, 0);
    const auto[loopStartLocation, pastEndLocation] = song->getLoopStartAndPastEndPositions(subsongId);
    songPlayer.play(startLocation, startLocation, pastEndLocation, true, true);

    // Reads all the ticks of the Song.
    auto emptyIterationCount = 0U;
    auto progressPercent = -1;
    const auto psgCount = song->getPsgCount(startLocation.getSubsongId());

    for (auto iterationIndex = 0; iterationIndex < iterationCount; ++iterationIndex) {
        // Calls the player. It will in turn calls our callback for the event!
        // All the PSGs must be read.
        for (auto psgIndex = 0; psgIndex < psgCount; ++psgIndex) {
            const auto [psgRegisters, _] = songPlayer.getNextRegisters(psgIndex);
            // Only processes the first PSG, ignores the other PSGs.
            if (psgIndex != 0) {
                continue;
            }

            const auto eventNumber = psgRegisters->getEventNumber();
            const auto mustBeEncoded = determineIfEventMustBeEncoded(eventNumber);

            // Is there a loop here? If yes, encodes the possible wait first, then the label for the loop.
            if (iterationIndex == loopToCounter) {
                // Encodes the possible wait, if any.
                if (emptyIterationCount > 0) {
                    encodeEvent(sourceGenerator, emptyIterationCount, 0);
                    emptyIterationCount = 0;
                }
                // Encodes the loop label.
                sourceGenerator.declareLabel(getLoopLabel());
            }

            // Should an event be encoded?
            if (mustBeEncoded) {
                // There is an event. Encodes the waiting plus the event.
                encodeEvent(sourceGenerator, emptyIterationCount, eventNumber);
                emptyIterationCount = 0;
            } else {
                // No event? Increases the waiting counter.
                ++emptyIterationCount;
            }

            // Notifies for the progress, if the percent value has change (to avoid sending too many events).
            if (const auto newProgressPercent = static_cast<int>(static_cast<double>(iterationIndex) / iterationCount * 100); progressPercent != newProgressPercent) {
                progressPercent = newProgressPercent;
                onTaskProgressed(progressPercent, 100);
            }
        }
    }

    // In the end, encodes the last "wait" so that the full song is encoded.
    encodeEvent(sourceGenerator, emptyIterationCount, 0);
    encodeEndEvent(sourceGenerator);

    sourceGenerator.declareByteRegionEnd();

    sourceGenerator.declareEndOfFile();

    exportResult->setSongStream(std::move(songOutputStream));

    return { true, std::move(exportResult) };
}

void EventsExporter::encodeHeader(SourceGenerator& sourceGenerator) const noexcept
{
    sourceGenerator.declareComment("Events generated by Arkos Tracker 3.").addEmptyLine();

    // Adds an ORG?
    if (exportConfiguration.getAddress().isPresent()) {
        sourceGenerator.declareAddressChange(exportConfiguration.getAddress().getValue()).addEmptyLine();
    }

    // Encodes the Events label.
    const auto firstLabel = getEventsLabel();
    sourceGenerator.declareLabel(firstLabel);
    sourceGenerator.declareExternalLabel(firstLabel);
}

void EventsExporter::encodeEvent(SourceGenerator& sourceGenerator, unsigned int duration, int eventNumber) noexcept
{
    constexpr auto maximumDuration = 0xfffeU;

    // Encodes the exceeding part (unlikely!), using a event '0'.
    while (duration > 0xfffe) {
        // The duration encoded is always +1.
        sourceGenerator.declareWord(maximumDuration + 1, "16 bits is not enough. Wait.");
        sourceGenerator.declareByte(0);

        duration -= maximumDuration;
    }

    jassert(duration < maximumDuration);

    // Encodes the "normal" part. The duration encoded is always +1.
    sourceGenerator.declareWord(static_cast<int>(duration + 1U), "Wait for " + juce::String(duration) + " frames.");
    sourceGenerator.declareByte(eventNumber);

    sourceGenerator.addEmptyLine();
}

void EventsExporter::encodeEndEvent(SourceGenerator& sourceGenerator) const noexcept
{
    sourceGenerator.declareWord(0, "End of sequence.");
    sourceGenerator.declareWordForceReference(getLoopLabel(), "Loops here.");
}

void EventsExporter::fillSampleIndexes() noexcept
{
    jassert(sampleIndexes.empty());     // Called twice?

    // Finds all the sample instruments.
    song->performOnConstInstruments([&] (const std::vector<std::unique_ptr<Instrument>>& instruments) {
        auto index = 0;
        for (const auto& instrument : instruments) {
            if (instrument->getType() == InstrumentType::sampleInstrument) {
                sampleIndexes.emplace(index);
            }
            ++index;
        }
    });
}

bool EventsExporter::determineIfEventMustBeEncoded(const int eventNumber) const noexcept
{
    if (eventNumber <= 0) {
        return false;
    }

    const auto isSampleInTypesToExport = (typesToExport.find(Type::samples) != typesToExport.cend());
    const auto isEventInTypesToExport = (typesToExport.find(Type::events) != typesToExport.cend());

    if (isSampleInTypesToExport && (sampleIndexes.find(eventNumber) != sampleIndexes.cend())) {
        return true;
    }
    if (isEventInTypesToExport && (sampleIndexes.find(eventNumber) == sampleIndexes.cend())) {
        return true;
    }

    return false;
}

}   // namespace arkostracker
