#include "AkmExporter.h"

#include "../../business/song/tool/optimizers/SongOptimizer.h"
#include "../sourceGenerator/SourceGenerator.h"
#include "process/AkmSongEncoder.h"
#include "process/AkmSubsongEncoder.h"

namespace arkostracker 
{

AkmExporter::AkmExporter(const Song& pOriginalSong, ExportConfiguration pExportConfiguration) noexcept :
        originalSong(pOriginalSong),
        song(),
        exportConfiguration(std::move(pExportConfiguration))
{
}

std::pair<bool, std::unique_ptr<SongExportResult>> AkmExporter::performTask() noexcept
{
    jassert(song == nullptr);       // Called twice?

    auto exportResult = std::make_unique<SongExportResult>();

    // Full optimization.
    song = SongOptimizer::optimize(originalSong, exportConfiguration);
    if (song == nullptr) {
        exportResult->addError(juce::translate("Too many inline arpeggios."));  // So far that's the only thing that could prevent optimization.
        return { false, std::move(exportResult) };
    }

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    sourceGenerator.setPrefixForDisark(exportConfiguration.getBaseLabel());
    const auto songName = song->getName();
    sourceGenerator.declareComment((songName.isEmpty() ? "" : (songName + ", ")) + "AKM format, v1.0.").addEmptyLine();
    sourceGenerator.declareComment("Generated by Arkos Tracker 3.").addEmptyLine();

    // Encodes the Song part.
    AkmSongEncoder akmSongEncoder(*song, exportConfiguration, *exportResult, getBaseLabel(),
                                  [this](const int subsongIndex) { return getBaseLabel(subsongIndex); });
    akmSongEncoder.encodeSong();

    // Encodes each Subsong.
    for (const auto& subsongId : song->getSubsongIds()) {
        const auto subsongIndex = song->getSubsongIndex(subsongId);
        if (subsongIndex.isAbsent()) {
            exportResult->addError("Subsong could not be reached!");
            continue;
        }

        AkmSubsongEncoder akmSubsongEncoder(*song, subsongId, exportConfiguration, *exportResult,
                                            getBaseLabel(subsongIndex.getValue()));

        // Generates the structures. It may fail (song too complex).
        akmSubsongEncoder.generateSubsong();
    }

    const auto success = exportResult->isOk();
    return { success, std::move(exportResult) };
}

juce::String AkmExporter::getBaseLabel(const int subsongIndex) const noexcept
{
    if (subsongIndex < 0) {
        return exportConfiguration.getBaseLabel();
    }
    return exportConfiguration.getBaseLabel() + "Subsong" + juce::String(subsongIndex);
}

}   // namespace arkostracker
