#include "AkgExporter.h"

#include "../../business/instrument/SampleResampler.h"
#include "../../business/song/tool/browser/CellBrowser.h"
#include "../../business/song/tool/optimizers/SongOptimizer.h"
#include "../samples/SampleCodeGenerator.h"
#include "../sourceGenerator/SourceGenerator.h"
#include "process/AkgSubsongExporter.h"
#include "process/PsgInstrumentCellEncoder.h"

namespace arkostracker 
{

const size_t AkgExporter::fastEffectCount = 128U;

const juce::String AkgExporter::startLabelString = "Start";                           // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::arpeggioTableLabelString = "ArpeggioTable";           // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::arpeggioLabelString = "Arpeggio";                     // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::pitchTableLabelString = "PitchTable";                 // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::pitchLabelString = "Pitch";                           // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::instrumentTableLabelString = "InstrumentTable";       // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::instrumentLabelString = "Instrument";                 // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::emptyInstrumentLabel = "EmptyInstrument";             // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::effectBlockTableLabelString = "EffectBlockTable";     // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::effectBlockLabelString = "EffectBlock";               // NOLINT(cert-err58-cpp, *-statically-constructed-objects)
const juce::String AkgExporter::loopSuffixLabelString = "_Loop";                      // NOLINT(cert-err58-cpp, *-statically-constructed-objects)

AkgExporter::AkgExporter(const Song& pOriginalSong, ExportConfiguration pExportConfiguration) noexcept:
        originalSong(pOriginalSong),
        song(),
        exportConfiguration(std::move(pExportConfiguration)),
        effectBlocks(),
        effectBlockIdToReferenceCounter(),
        effectBlockIdToIndex(),
        indexToEffectBlockId(),
        effectBlockIdToOffsetFromEffectBlockTable()
{
    jassert(!exportConfiguration.getSubsongIds().empty());
    jassert(!exportConfiguration.mustEncodeAllAddressesAsRelativeToSongStart());      // Unused.
}

std::pair<bool, std::unique_ptr<SongExportResult>> AkgExporter::performTask() noexcept
{
    jassert(song == nullptr);       // Called twice?

    auto exportResult = std::make_unique<SongExportResult>();

    // Full optimization.
    song = SongOptimizer::optimize(originalSong, exportConfiguration);
    if (song == nullptr) {
        exportResult->addError(juce::translate("Too many inline arpeggios."));  // So far that's the only thing that could prevent optimization.
        return { false, std::move(exportResult) };
    }

    auto songOutputStream = std::make_unique<juce::MemoryOutputStream>();
    SourceGenerator sourceGenerator(exportConfiguration.getSourceConfiguration(), *songOutputStream);

    sourceGenerator.setPrefixForDisark(exportConfiguration.getBaseLabel());
    const auto songName = song->getName();
    sourceGenerator.declareComment((songName.isEmpty() ? "" : (songName + ", ")) + "AKG format, v1.0.").addEmptyLine();
    sourceGenerator.declareComment("Generated by Arkos Tracker 3.").addEmptyLine();

    // Org?
    if (exportConfiguration.getAddress().isPresent()) {
        sourceGenerator.declareAddressChange(exportConfiguration.getAddress().getValue()).addEmptyLine().addEmptyLine();
    }

    // Encodes the metadata of the Song.
    const auto startLabel = getBaseLabel() + startLabelString;
    sourceGenerator.declareLabel(startLabel);
    sourceGenerator.declareExternalLabel(startLabel);

    // The Song is a byte area by default. No reference will be inferred.
    sourceGenerator.declareByteRegionStart();

    sourceGenerator.declareString("AT30");

    // Declares the references to the Arpeggio, Pitch and Instrument tables.
    sourceGenerator.declarePointerRegionStart();
    sourceGenerator.declareWord(getArpeggioTableLabel(), "The address of the Arpeggio table.");
    sourceGenerator.declareWord(getPitchTableLabel(), "The address of the Pitch table.");
    sourceGenerator.declareWord(getInstrumentTableLabel(), "The address of the Instrument table.");
    sourceGenerator.declareWord(getEffectBlockTableLabel(), "The address of the Effect Block table.");
    sourceGenerator.declarePointerRegionEnd();

    sourceGenerator.addEmptyLine().addEmptyLine().declareComment("The addresses of each Subsong.");

    // Encodes the tables of every Subsong.
    sourceGenerator.declarePointerRegionStart();
    for (const auto& subsongId : exportConfiguration.getSubsongIds()) {
        const auto subsongIndex = song->getSubsongIndex(subsongId);
        if (subsongIndex.isAbsent()) {
            jassertfalse;           // Should never happen!
            return { false, nullptr };
        }
        sourceGenerator.declareWord(getBaseLabel(subsongIndex.getValue()) + startLabelString);
    }
    sourceGenerator.declarePointerRegionEnd().addEmptyLine();

    PlayerConfiguration playerConfiguration;

    // Encodes the Arpeggios and Pitches.
    auto success = encodeExpressionTableAndExpressions(sourceGenerator, true);
    success = success && encodeExpressionTableAndExpressions(sourceGenerator, false);

    // Encodes the Instruments.
    success = success && encodeInstrumentTableAndInstruments(sourceGenerator, playerConfiguration);

    if (!success) {
        return { false, nullptr };
    }

    // Encodes the Effect Blocks.
    generateEffectBlocks(playerConfiguration);
    encodeEffectBlockIndexesTable(sourceGenerator);
    encodeEffectBlocks(sourceGenerator);
    sourceGenerator.addEmptyLine();
    buildEffectBlocksOffset();

    sourceGenerator.declareByteRegionEnd();
    sourceGenerator.addEmptyLine().declareEndOfFile();

    // Encodes each Subsong. Each will fill the Export Result.
    for (const auto& subsongId : exportConfiguration.getSubsongIds()) {
        const auto subsongIndex = song->getSubsongIndex(subsongId).getValue();
        AkgSubsongExporter subsongExporter(*song, subsongId, exportConfiguration, playerConfiguration, *exportResult,
                                           getBaseLabel(subsongIndex), effectBlockIdToIndex, effectBlockIdToOffsetFromEffectBlockTable);
        subsongExporter.generateSubsong();
    }

    exportResult->setSongStream(std::move(songOutputStream));
    exportResult->setPlayerConfiguration(playerConfiguration);

    return { success, std::move(exportResult) };
}

juce::String AkgExporter::getBaseLabel(const int subsongIndex) const noexcept
{
    if (subsongIndex < 0) {
        return exportConfiguration.getBaseLabel();
    }
    return exportConfiguration.getBaseLabel() + "Subsong" + juce::String(subsongIndex) + "_";
}

juce::String AkgExporter::getArpeggioTableLabel() const noexcept
{
    return getBaseLabel(-1) + arpeggioTableLabelString;
}

juce::String AkgExporter::getArpeggioLabel(const int arpeggioIndex) const noexcept
{
    jassert(arpeggioIndex > 0);

    return getBaseLabel(-1) + arpeggioLabelString + juce::String(arpeggioIndex);
}

juce::String AkgExporter::getPitchTableLabel() const noexcept
{
    return getBaseLabel(-1) + pitchTableLabelString;
}

juce::String AkgExporter::getPitchLabel(const int pitchIndex) const noexcept
{
    jassert(pitchIndex > 0);

    return getBaseLabel(-1) + pitchLabelString + juce::String(pitchIndex);
}

juce::String AkgExporter::getInstrumentTableLabel() const noexcept
{
    return getBaseLabel(-1) + instrumentTableLabelString;
}

juce::String AkgExporter::getEffectBlockTableLabel() const noexcept
{
    return getBaseLabel(-1) + effectBlockTableLabelString;
}

juce::String AkgExporter::getInstrumentLabel(const int instrumentIndex) const noexcept
{
    // The "empty" Instrument has a specific Label.
    return getBaseLabel(-1) + ((instrumentIndex == 0) ? emptyInstrumentLabel :
                               (instrumentLabelString + juce::String(instrumentIndex)));
}

juce::String AkgExporter::getEffectBlockLabel(const EncodedEffectBlock& encodedEffectBlock) const noexcept
{
    return getEffectBlockLabel(encodedEffectBlock.getId());
}

juce::String AkgExporter::getEffectBlockLabel(const juce::String& encodedEffectBlockId) const noexcept
{
    return getBaseLabel(-1) + effectBlockLabelString + "_" + encodedEffectBlockId;
}

bool AkgExporter::encodeExpressionTableAndExpressions(SourceGenerator& sourceGenerator, const bool isArpeggio) const noexcept
{
    const auto count = song->getExpressionHandler(isArpeggio).getCount();

    // Encodes the start of the table.
    sourceGenerator.declareComment(isArpeggio ? "Declares all the Arpeggios." : "Declares all the Pitches.");
    sourceGenerator.declareLabel(isArpeggio ? getArpeggioTableLabel() : getPitchTableLabel());

    sourceGenerator.declarePointerRegionStart();
    constexpr auto firstExpressionIndex = 1;
    for (auto expressionIndex = firstExpressionIndex; expressionIndex < count; ++expressionIndex) {
        sourceGenerator.declareWord(isArpeggio ? getArpeggioLabel(expressionIndex) : getPitchLabel(expressionIndex));
    }
    sourceGenerator.declarePointerRegionEnd();

    // Encodes the Expressions.
    auto success = true;
    for (auto expressionIndex = firstExpressionIndex; expressionIndex < count; ++expressionIndex) {
        success = success && encodeExpression(sourceGenerator, isArpeggio, expressionIndex);
    }

    sourceGenerator.addEmptyLine();

    return success;
}

bool AkgExporter::encodeExpression(SourceGenerator& sourceGenerator, const bool isArpeggio, const int expressionIndex) const noexcept
{
    const auto& expressionHandler = song->getExpressionHandler(isArpeggio);

    // Encodes the label.
    const auto expressionLabel = isArpeggio ? getArpeggioLabel(expressionIndex) : getPitchLabel(expressionIndex);
    sourceGenerator.declareLabel(expressionLabel);

    const auto expressionId = expressionHandler.find(expressionIndex);
    if (expressionId.isAbsent()) {
        jassertfalse;       // Should never happen!
        return false;
    }
    const auto expression = expressionHandler.getExpressionCopy(expressionId.getValueRef());

    const auto speed = expression.getSpeed();
    const auto loop = expression.getLoop(true);
    const auto loopToIndex = loop.getStartIndex();
    const auto endIndex = loop.getEndIndex();

    sourceGenerator.declareByte(getIncreasedSpeed(speed), "The speed.").addEmptyLine();

    // Encodes each Cell.
    for (auto lineIndex = 0; lineIndex <= endIndex; ++lineIndex) {
        const auto value = expression.getValue(lineIndex, true);

        // The encoding is different from the Expression type.
        if (isArpeggio) {
            sourceGenerator.declareByte(value);
        } else {
            sourceGenerator.declareWord(-value);     // Encodes a PERIOD pitch, so invert the value.
            // Declares the loop or goto next.
            if (lineIndex == endIndex) {
                // Declares the loop.
                sourceGenerator.declareWordForceReference(expressionLabel + " + 4 * " + juce::String(loopToIndex) + " + 1");
            } else {
                // Creates a label for each line (request from a user to avoid the user of $).
                const auto pitchLineLabel = expressionLabel + "_GotoNextForLine" + juce::String(lineIndex);
                sourceGenerator.declareLabel(pitchLineLabel);
                // Goto next item.
                sourceGenerator.declareWordForceReference(pitchLineLabel + " + 2");
            }
        }

    }

    if (isArpeggio) {
        // Declares the loop (-128 followed by the label to jump (that is, the loop index plus 1 to skip the header).
        sourceGenerator.declareByte(-128)
                .declareWordForceReference(expressionLabel + " + " + juce::String(loopToIndex) + " + 1", "Loop.");
    }

    sourceGenerator.addEmptyLine();

    return true;
}

int AkgExporter::getIncreasedSpeed(int speed) noexcept
{
    return (++speed == 256) ? 0 : speed;
}

bool AkgExporter::encodeInstrumentTableAndInstruments(SourceGenerator& sourceGenerator, PlayerConfiguration& playerConfiguration) const noexcept
{
    const auto instrumentIds = song->getInstrumentIds();

    // Encodes the start of the table.
    sourceGenerator.declareComment("Declares all the Instruments.");
    sourceGenerator.declareLabel(getInstrumentTableLabel());
    sourceGenerator.declarePointerRegionStart();

    // Note that even the empty Instrument is declared and encoded!
    const auto areSamplesExported = exportConfiguration.getSampleEncoderFlags().areSamplesExported();
    auto instrumentIndex = 0;
    for (const auto& instrumentId : instrumentIds) {
        song->performOnInstrument(instrumentId, [&](const Instrument& instrument) {
            const auto instrumentType = instrument.getType();
            // PSG Instruments are always exported, what about the Sample Instruments?
            const auto encodeInstrument = (instrumentType == InstrumentType::psgInstrument)
                    || ((instrumentType == InstrumentType::sampleInstrument) && areSamplesExported);

            if (encodeInstrument) {
                sourceGenerator.declareWord(getInstrumentLabel(instrumentIndex));
            } else {
                sourceGenerator.declareWord(0, "Not exported.");
            }
        });
        ++instrumentIndex;
    }
    sourceGenerator.declarePointerRegionEnd();
    sourceGenerator.addEmptyLine();

    // Encodes the Instruments.
    instrumentIndex = 0;
    auto success = true;
    for (const auto& instrumentId : instrumentIds) {
        song->performOnInstrument(instrumentId, [&](const Instrument& instrument) noexcept {
            const auto instrumentType = instrument.getType();
            if (instrumentType == InstrumentType::psgInstrument) {
                encodePsgInstrument(sourceGenerator, playerConfiguration, instrumentIndex, instrument);
            } else if ((instrumentType == InstrumentType::sampleInstrument) && areSamplesExported) {
                // Encodes the Sample Instrument.
                success = success && encodeSampleInstrument(sourceGenerator, instrumentIndex, instrument);
            }
        });

        ++instrumentIndex;
    }

    sourceGenerator.addEmptyLine();

    return success;     // Fails if samples >64kb.
}

void AkgExporter::encodePsgInstrument(SourceGenerator& sourceGenerator, PlayerConfiguration& playerConfiguration,
                                      const int instrumentIndex, const Instrument& instrument) const noexcept
{
    jassert(instrument.getType() == InstrumentType::psgInstrument);
    const auto& psgPart = instrument.getConstPsgPart();
    jassert(!psgPart.isInstrumentRetrig());        // The Retrig has been removed (transferred to the Instrument Cells).

    PsgInstrumentCellEncoder encoder(sourceGenerator);

    // Encodes the label.
    sourceGenerator.declareLabel(getInstrumentLabel(instrumentIndex));

    // Encodes the metadata.
    sourceGenerator.declareByte(getIncreasedSpeed(psgPart.getSpeed()), "The speed (>0, 0 for 256).").addEmptyLine();

    // Encodes the reference to the cells, from a generated label from the Cells.
    const auto& loop = psgPart.getMainLoopRef();
    // If instrument is the "empty" one, we consider it is NOT looping.
    // A "loop to silence" is encoded (optimization, and allows the PlayerConfiguration to have only non-looping instruments).
    const auto isLooping = loop.isLooping() && (instrumentIndex != 0);
    const auto labelInstrumentToLoopTo = getInstrumentLabel(isLooping ? instrumentIndex : 0) + loopSuffixLabelString;
    const auto loopStartIndex = loop.getStartIndex();

    playerConfiguration.addFlag(isLooping, PlayerConfigurationFlag::instrumentLoopTo);

    // Encodes each Cell.
    for (auto cellIndex = 0, lastCellIndex = loop.getEndIndex(); cellIndex <= lastCellIndex; ++cellIndex) {
        // Should a loop be encoded here? Encodes the label if yes.
        if (isLooping && (cellIndex == loopStartIndex)) {
            sourceGenerator.declareLabel(labelInstrumentToLoopTo);
        }

        encoder.encodePsgInstrumentCell(playerConfiguration, psgPart, cellIndex);
    }

    // Encodes the loop.
    if (isLooping) {
        encoder.encodeLoopToLabel(labelInstrumentToLoopTo);
    } else {
        // Loops to "silent" instrument.
        encoder.encodeLoopToSilence();
    }

    sourceGenerator.addEmptyLine();
}

void AkgExporter::generateEffectBlocks(PlayerConfiguration& playerConfiguration) noexcept
{
    effectBlocks.clear();
    effectBlockIdToReferenceCounter.clear();

    // Browses each Subsong, looking for all the effects.
    CellBrowser cellBrowser(*song);
    for (const auto& subsongId : exportConfiguration.getSubsongIds()) {
        cellBrowser.browseSubsongCells(true, subsongId, [&](const Cell& cell) {
            // Any effects on each Cell?
            if (cell.hasEffects()) {
                // Gets the effects, encodes them, stores them.
                const auto note = cell.getNote();
                const auto& cellEffects = cell.getEffects();
                const EncodedEffectBlock encodedEffectBlock(playerConfiguration, note, cellEffects);

                // If unique, adds the effect block.
                if (std::find(effectBlocks.cbegin(), effectBlocks.cend(), encodedEffectBlock) == effectBlocks.cend()) {
                    effectBlocks.push_back(encodedEffectBlock);
                }

                // Increases the reference counter to this Effect Block.
                increaseEffectBlockReferenceCounter(encodedEffectBlock);
            }

            return nullptr;     // We don't want to modify them.
        });
    }

    buildEffectBlockWeight();
}

void AkgExporter::increaseEffectBlockReferenceCounter(const EncodedEffectBlock& encodedEffectBlock) noexcept
{
    const auto effectBlockId = encodedEffectBlock.getId();

    // If the ID already exists, increases the reference counter.
    if (const auto iterator = effectBlockIdToReferenceCounter.find(effectBlockId); iterator == effectBlockIdToReferenceCounter.cend()) {
        // Does not exist. Adds it with a unique reference.
        effectBlockIdToReferenceCounter.insert(std::make_pair(effectBlockId, 1));
    } else {
        // Does exist. Only increases the reference counter.
        auto& referenceCounter = iterator->second;
        ++referenceCounter;
    }
}

void AkgExporter::buildEffectBlocksOffset() noexcept
{
    jassert(effectBlockIdToReferenceCounter.size() == effectBlocks.size());
    jassert(std::min(fastEffectCount, effectBlockIdToReferenceCounter.size()) == effectBlockIdToIndex.size());  // Only 128 max are encoded.
    jassert(std::min(fastEffectCount, effectBlockIdToReferenceCounter.size()) == indexToEffectBlockId.size());
    effectBlockIdToOffsetFromEffectBlockTable.clear();

    // The offset first includes the table itself, which can have 128 entries (of word) max.
    auto offset = static_cast<int>(std::min(effectBlockIdToIndex.size(), fastEffectCount) * 2U);

    // Browses each EffectBlock, as they are encoded, thus increasing the offset. Each EffectBlock can be given its offset from the EffectBlock Table.
    for (const auto& effectBlock : effectBlocks) {
        effectBlockIdToOffsetFromEffectBlockTable.insert(std::make_pair(effectBlock.getId(), offset));

        offset += effectBlock.getByteCount();
    }
}

void AkgExporter::buildEffectBlockWeight() noexcept
{
    jassert(effectBlocks.size() == effectBlockIdToReferenceCounter.size());

    // Stores the ids of the Effect blocks, from "heaviest/most used" to less used.
    std::vector<EffectBlockIdAndWeight> heaviestEffectBlockIdsAndWeights;

    for (auto& effectBlock : effectBlocks) {
        // How many references to it? There should be at least one.
        const auto& id = effectBlock.getId();

        auto iterator = effectBlockIdToReferenceCounter.find(id);
        jassert(iterator != effectBlockIdToReferenceCounter.cend());        // There should be one!
        const auto iterationCount = iterator->second;

        // The weight is an approximation, because if encoded as an index, it will cost less, and more if encoded as a weight!
        const auto weight = iterationCount * effectBlock.getByteCount();

        heaviestEffectBlockIdsAndWeights.emplace_back(id, weight);
    }

    // Sorts the Blocks.
    constexpr SortHeaviestEffectBlockIdsPredicate sortHeaviestEffectBlockIdsPredicate;
    std::sort(heaviestEffectBlockIdsAndWeights.begin(), heaviestEffectBlockIdsAndWeights.end(), sortHeaviestEffectBlockIdsPredicate);

    // Only keeps the 128 first.
    if (heaviestEffectBlockIdsAndWeights.size() > fastEffectCount) {
        heaviestEffectBlockIdsAndWeights.resize(fastEffectCount);
    }

    // Only keeps the IDs.
    auto index = 0;
    for (auto& item : heaviestEffectBlockIdsAndWeights) {
        const auto& id = item.getId();
        effectBlockIdToIndex.emplace(id, index++);
        indexToEffectBlockId.push_back(id);
    }
}

void AkgExporter::encodeEffectBlockIndexesTable(SourceGenerator& sourceGenerator) const noexcept
{
    // Encodes the label of the Table in all cases.
    sourceGenerator.declareComment("The indexes of the effect blocks used by this song.");
    sourceGenerator.declareLabel(getEffectBlockTableLabel());

    // Stops if there is nothing to put inside the table.
    if (indexToEffectBlockId.empty()) {
        return;
    }

    sourceGenerator.declarePointerRegionStart();

    // Encodes the addresses of all the EffectBlocks that are index-referenced.
    auto index = 0;
    for (const auto& effectBlockId : indexToEffectBlockId) {
        sourceGenerator.declareWord(getEffectBlockLabel(effectBlockId), "Index " + juce::String(index));
        ++index;
    }
    sourceGenerator.declarePointerRegionEnd();
}

void AkgExporter::encodeEffectBlocks(SourceGenerator& sourceGenerator) const noexcept
{
    // The effects blocks can be encoded directly.
    for (const auto& effectBlock : effectBlocks) {
        encodeEffectBlock(sourceGenerator, effectBlock);
    }
}

void AkgExporter::encodeEffectBlock(SourceGenerator& sourceGenerator, const EncodedEffectBlock& encodedEffectBlock) const noexcept
{
    // Generates the label, directly from the content of the block.
    const auto encodedBytes = encodedEffectBlock.getBytes();

    sourceGenerator.declareLabel(getEffectBlockLabel(encodedEffectBlock));

    // Encodes in a raw manner.
    for (const auto c : encodedBytes) {
        sourceGenerator.declareByte(c);
    }
    sourceGenerator.addEmptyLine();
}

bool AkgExporter::encodeSampleInstrument(SourceGenerator& sourceGenerator, const int instrumentIndex, const Instrument& instrument) const noexcept
{
    // Encodes the Sample itself.
    const auto sampleLabel = getInstrumentLabel(instrumentIndex);       // Important to have the same label as the normal Instruments, as they are referenced in the table index.
    sourceGenerator.declareLabel(sampleLabel);

    // Encodes the header.
    sourceGenerator.declareByte(0, "Speed (useless for samples).");
    sourceGenerator.declareByte(0b11111110, "Tag to recognize the Sample Instrument (sound end without loop, with b7-b3 to 1).");

    // Encodes the Sample itself. Important to resample before, it is not done automatically.
    const auto& samplePart = SampleResampler::resample(instrument.getConstSamplePart());

    sourceGenerator.declareComment("The sample data.");
    const auto success = SampleCodeGenerator::generateHeader(sourceGenerator, exportConfiguration.getSampleEncoderFlags(), samplePart);
    // Are the samples too big?
    if (!success) {
        return false;
    }
    SampleCodeGenerator::generateSampleData(sourceGenerator, exportConfiguration.getSampleEncoderFlags(), samplePart);

    sourceGenerator.addEmptyLine();

    return true;
}

}   // namespace arkostracker
