#include "Tool.h"

#include "../../ProjectInfo.h"
#include "../../export/fap/FapExporter.h"
#include "../../utils/FileUtil.h"
#include "../utils/CommandLineToolHelper.h"

namespace arkostracker
{

int Tool::execute(const int argc, char* argv[])      // NOLINT(*-avoid-c-arrays,clion-misra-cpp2008-3-1-3)
{
    const auto guiInit = CommandLineToolHelper::initJuce();

    const auto fapVersion = projectinfo::fapVersion;

    // Creates the command line.
    const auto description = juce::String("Converts to FAP any song that can be loaded into Arkos Tracker 3. FAP version: ") + fapVersion + ".\n"
                                     "Usage: SongToFap " + CommandLineToolHelper::getDescriptionForSubsongParameter() +
                                     " [-c] <path to input song> <path to output FAP>";
    std::vector<CommandLineArgumentDescriptor*> descriptors;

    CommandLineToolHelper commandLineToolHelper;

    // Uses the helper to declare common parameters.
    commandLineToolHelper.declareSubsongParameter(descriptors);
    commandLineToolHelper.declareInputSongParameter(descriptors);

    auto descriptorParameterOutput = CommandLineArgumentDescriptor::buildArgumentWithDirectValue(
            juce::translate("<path to output FAP>"), juce::translate("Path and filename to the FAP file to create."), true);

    descriptors.push_back(&descriptorParameterOutput);

    // Starts parsing.
    const auto parseResult = commandLineToolHelper.parseAndGetSong(argc, argv, descriptors, description);
    if (parseResult.second == nullptr) {
        return parseResult.first ? 0 : -1;
    }
    const auto song = parseResult.second;

    // Reads the possible Subsong and PSG index.
    const auto subsongId = commandLineToolHelper.getSubsongIdOrWriteError(*song);
    if (subsongId.isAbsent()) {
        return -1;
    }

    // Makes the export.
    FapExporter exporter(song, subsongId.getValue());
    const auto& [fapSuccess, fapResult] = exporter.performTask();
    const auto& [fapData, bufferSize, playTimeInNops, registerCountToPlay, isR12Constant] = *fapResult;
    if (!fapSuccess) {
        CommandLineToolHelper::cerr(juce::translate("Export to FAP failed!"));
        return -1;
    }
    if (bufferSize <= 0) {
        CommandLineToolHelper::cerr(juce::translate("Buffer size is illegal, export failed!"));
        return -1;
    }

    const auto outputFileString = descriptorParameterOutput.getDirectValue();
    const auto success = FileUtil::saveMemoryBlockToFile(outputFileString, fapData);
    if (!success) {
        CommandLineToolHelper::cout(juce::translate("Saving failed!"));
        return -1;
    }

    CommandLineToolHelper::cout(juce::translate("Export to FAP successful."));
    return 0;
}

}   // namespace arkostracker
