#include "SampleResampler.h"

#include "../../player/TemperedScaleUtil.h"
#include "../../utils/NumberUtil.h"

namespace arkostracker
{

SamplePart SampleResampler::resample(const SamplePart& inputSamplePart) noexcept
{
    // If the same diginote, returns the same data (though copied...).
    const auto newDigidrumNote = inputSamplePart.getDigidrumNote();
    if (newDigidrumNote == PsgValues::digidrumNote) {
        return inputSamplePart;
    }

    constexpr auto referenceFrequency = 440.0F;     // Doesn't matter in this case.

    const auto originalFrequency = TemperedScaleUtil::getFrequency(PsgValues::digidrumNote, referenceFrequency);
    const auto newFrequency = TemperedScaleUtil::getFrequency(newDigidrumNote, referenceFrequency);

    const auto sampleStep = static_cast<double>(newFrequency) / static_cast<double>(originalFrequency);

    const auto& inputSample = inputSamplePart.getSample();
    const auto& inputSampleData = inputSample->getData();
    const auto inputSampleSize = inputSample->getLength();

    juce::MemoryBlock newSampleData;

    // Performs the resampling.
    char appendedData[1];
    auto index = 0.0;
    while (index < static_cast<double>(inputSampleSize)) {
        const auto readSample = inputSampleData[static_cast<int>(index)];
        appendedData[0] = readSample;
        newSampleData.append(appendedData, 1);
        index += sampleStep;
    }

    const auto inputLoop = inputSamplePart.getLoop();
    const auto newSample = std::make_shared<Sample>(newSampleData);
    const auto newSampleLastIndex = static_cast<int>(newSampleData.getSize()) - 1;
    // For the end, makes sure we go past the original index to reach the "final" sample.
    const auto newEndIndexDouble = std::ceil((inputLoop.getEndIndex() + 1) / sampleStep);
    // Added a security to avoid going over the bounds.
    const auto newEndIndex = NumberUtil::correctNumber(static_cast<int>(newEndIndexDouble) - 1, 0, newSampleLastIndex);
    const auto newStartIndex = NumberUtil::correctNumber(static_cast<int>(inputLoop.getStartIndex() / sampleStep), 0, newEndIndex);
    const auto newLoop = Loop(newStartIndex, newEndIndex, inputLoop.isLooping());

    return SamplePart(newSample, newLoop, inputSamplePart.getAmplificationRatio(), inputSamplePart.getFrequencyHz(),
        PsgValues::digidrumNote, inputSamplePart.getOriginalFileName());
}

}   // namespace arkostracker
